;;;; the VOPs and other necessary machine specific support
;;;; routines for call-out to C

;;;; This software is part of the SBCL system. See the README file for
;;;; more information.
;;;;
;;;; This software is derived from the CMU CL system, which was
;;;; written at Carnegie Mellon University and released into the
;;;; public domain. The software is in the public domain and is
;;;; provided with absolutely no warranty. See the COPYING and CREDITS
;;;; files for more information.

(in-package "SB-VM")

;; The MOVE-ARG vop is going to store args on the stack for
;; call-out. These tn's will be used for that. move-arg is normally
;; used for things going down the stack but C wants to have args
;; indexed in the positive direction.

(defstruct (arg-state (:copier nil))
  (register-args 0)
  (xmm-args 0)
  (stack-frame-size 0))
(declaim (freeze-type arg-state))

;;; Cache for struct classification to avoid redundant computation.
;;; Bound in make-call-out-tns when processing struct return types.
(defvar *cached-struct-classification* nil)

(defconstant max-int-args #.(length *c-call-register-arg-offsets*))
(defconstant max-xmm-args #+win32 4 #-win32 8)

(defun int-arg (state prim-type reg-sc stack-sc)
  (let ((reg-args (max (arg-state-register-args state)
                       #+win32 (arg-state-xmm-args state))))
    (cond ((< reg-args max-int-args)
           (setf (arg-state-register-args state) (1+ reg-args))
           (make-wired-tn* prim-type reg-sc
                             (nth reg-args *c-call-register-arg-offsets*)))
          (t
           (let ((frame-size (arg-state-stack-frame-size state)))
             (setf (arg-state-stack-frame-size state) (1+ frame-size))
             (make-wired-tn* prim-type stack-sc frame-size))))))

(define-alien-type-method (integer :arg-tn) (type state)
  (if (alien-integer-type-signed type)
      (int-arg state 'signed-byte-64 signed-reg-sc-number signed-stack-sc-number)
      (int-arg state 'unsigned-byte-64 unsigned-reg-sc-number unsigned-stack-sc-number)))

(define-alien-type-method (system-area-pointer :arg-tn) (type state)
  (declare (ignore type))
  (int-arg state 'system-area-pointer sap-reg-sc-number sap-stack-sc-number))

(defun float-arg (state prim-type reg-sc stack-sc)
  (let ((xmm-args (max (arg-state-xmm-args state)
                        #+win32 (arg-state-register-args state))))
    (cond ((< xmm-args max-xmm-args)
           (setf (arg-state-xmm-args state) (1+ xmm-args))
           (make-wired-tn* prim-type reg-sc
                             (nth xmm-args *float-regs*)))
          (t
           (let ((frame-size (arg-state-stack-frame-size state)))
             (setf (arg-state-stack-frame-size state) (1+ frame-size))
             (make-wired-tn* prim-type stack-sc frame-size))))))

(define-alien-type-method (double-float :arg-tn) (type state)
  (declare (ignore type))
  (float-arg state 'double-float double-reg-sc-number double-stack-sc-number))

(define-alien-type-method (single-float :arg-tn) (type state)
  (declare (ignore type))
  (float-arg state 'single-float single-reg-sc-number single-stack-sc-number))

(defstruct (result-state (:copier nil))
  (num-results 0))
(declaim (freeze-type result-state))

(defun result-reg-offset (slot)
  (ecase slot
    (0 rax-offset)
    (1 rdx-offset)))

(define-alien-type-method (integer :result-tn) (type state)
  (let ((num-results (result-state-num-results state)))
    (setf (result-state-num-results state) (1+ num-results))
    (multiple-value-bind (ptype reg-sc)
        (if (alien-integer-type-signed type)
            (values 'signed-byte-64 signed-reg-sc-number)
            (values 'unsigned-byte-64 unsigned-reg-sc-number))
      (make-wired-tn* ptype reg-sc (result-reg-offset num-results)))))

(define-alien-type-method (integer :naturalize-gen) (type alien)
  (if (<= (alien-type-bits type) 32)
      (if (alien-integer-type-signed type)
          `(sign-extend ,alien ,(alien-type-bits type))
          `(logand ,alien ,(1- (ash 1 (alien-type-bits type)))))
      alien))

(define-alien-type-method (system-area-pointer :result-tn) (type state)
  (declare (ignore type))
  (let ((num-results (result-state-num-results state)))
    (setf (result-state-num-results state) (1+ num-results))
    (make-wired-tn* 'system-area-pointer sap-reg-sc-number
                      (result-reg-offset num-results))))

(define-alien-type-method (double-float :result-tn) (type state)
  (declare (ignore type))
  (let ((num-results (result-state-num-results state)))
    (setf (result-state-num-results state) (1+ num-results))
    (make-wired-tn* 'double-float double-reg-sc-number num-results)))

(define-alien-type-method (single-float :result-tn) (type state)
  (declare (ignore type))
  (let ((num-results (result-state-num-results state)))
    (setf (result-state-num-results state) (1+ num-results))
    (make-wired-tn* 'single-float single-reg-sc-number num-results)))

(define-alien-type-method (values :result-tn) (type state)
  (let ((values (alien-values-type-values type)))
    (when (> (length values) 2)
      (error "Too many result values from c-call."))
    (mapcar (lambda (type)
              (invoke-alien-type-method :result-tn type state))
            values)))

;;;; Struct Return-by-Value Support (System V AMD64 ABI)

#-win32
(defun classify-field-sysv-amd64 (type)
  "Classify a single field type for SysV AMD64 ABI.
   Returns :INTEGER, :DOUBLE, or :MEMORY."
  (cond
    ;; Check specific types first, before general type checks
    ((sb-alien::alien-integer-type-p type) :integer)
    ((sb-alien::alien-pointer-type-p type) :integer)
    ((sb-alien::alien-single-float-type-p type) :double)
    ((sb-alien::alien-double-float-type-p type) :double)
    ;; Arrays are classified by their element type
    ((sb-alien::alien-array-type-p type)
     (let ((element-type (sb-alien::alien-array-type-element-type type)))
       (classify-field-sysv-amd64 element-type)))
    ;; Nested struct - recursively classify and inherit eightbyte classes
    ((sb-alien::alien-record-type-p type)
     (let ((nested (classify-struct type)))
       (if (sb-alien::struct-classification-memory-p nested)
           :memory
           ;; Merge all slots from nested struct to get dominant class
           ;; e.g., struct { double d; } should contribute :double, not :integer
           (reduce #'merge-classes
                   (sb-alien::struct-classification-register-slots nested)
                   :initial-value :no-class))))
    ;; System-area-pointer (must come after array/record checks)
    ((typep type 'sb-alien::alien-system-area-pointer-type) :integer)
    (t :memory)))

;;; Merge two classes within an eightbyte per ABI rules
#-win32
(defun merge-classes (class1 class2)
  "Merge two classes within an eightbyte per ABI rules.
   INTEGER dominates SSE; MEMORY dominates everything."
  (cond
    ((eq class1 class2) class1)
    ((eq class1 :no-class) class2)
    ((eq class2 :no-class) class1)
    ((or (eq class1 :memory) (eq class2 :memory)) :memory)
    ((or (eq class1 :integer) (eq class2 :integer)) :integer)
    (t :double)))

;;; Main classification function for x86-64 System V AMD64 ABI
#-win32
(defun classify-struct (record-type)
  "Classify struct for x86-64 System V ABI return.
   Returns STRUCT-CLASSIFICATION."
  (let* ((bits (sb-alien::alien-type-bits record-type))
         (byte-size (ceiling bits 8))
         (alignment (sb-alien::alien-type-alignment record-type)))
    ;; Rule: Structs > 16 bytes always use memory (hidden pointer)
    (when (> byte-size 16)
      (return-from classify-struct
        (sb-alien::make-struct-classification
         :register-slots '(:memory)
         :size byte-size
         :alignment alignment
         :memory-p t)))

    ;; Classify each eightbyte
    (let* ((num-eightbytes (max 1 (ceiling byte-size 8)))
           (eightbytes (make-list num-eightbytes :initial-element :no-class)))
      ;; Iterate through fields and classify
      (dolist (field (sb-alien::alien-record-type-fields record-type))
        (let* ((field-offset-bits (sb-alien::alien-record-field-offset field))
               (field-type (sb-alien::alien-record-field-type field))
               (field-bits (sb-alien::alien-type-bits field-type))
               (field-offset-bytes (floor field-offset-bits 8))
               (field-size-bytes (ceiling field-bits 8))
               (field-class (classify-field-sysv-amd64 field-type)))
          ;; Apply class to all eightbytes this field spans
          (loop for byte-offset from field-offset-bytes below (+ field-offset-bytes field-size-bytes) by 8
                for eightbyte-index = (floor byte-offset 8)
                when (< eightbyte-index num-eightbytes)
                do (setf (nth eightbyte-index eightbytes)
                         (merge-classes (nth eightbyte-index eightbytes)
                                        field-class)))))

      ;; Post-merge cleanup per ABI: if second eightbyte is MEMORY, first must be too
      (when (and (> num-eightbytes 1)
                 (eq (second eightbytes) :memory))
        (setf (first eightbytes) :memory))

      ;; Convert remaining :no-class to :integer (padding bytes are treated as integer)
      (setf eightbytes
            (mapcar (lambda (c) (if (eq c :no-class) :integer c)) eightbytes))

      (sb-alien::make-struct-classification
       :register-slots eightbytes
       :size byte-size
       :alignment alignment
       :memory-p (member :memory eightbytes)))))

#+win32
(defun classify-struct (record-type)
  "Classify struct for Windows AMD64 ABI.
Size-based only: <=8 bytes in single integer register, >8 bytes via pointer.
Floats are passed in integer registers."
  (let* ((bits (sb-alien::alien-type-bits record-type))
         (byte-size (ceiling bits 8))
         (alignment (sb-alien::alien-type-alignment record-type)))
    (if (> byte-size 8)
        ;; Large struct: hidden pointer
        (sb-alien::make-struct-classification
         :register-slots '(:memory)
         :size byte-size
         :alignment alignment
         :memory-p t)
        ;; Small struct: single integer
        (sb-alien::make-struct-classification
         :register-slots '(:integer)
         :size byte-size
         :alignment alignment
         :memory-p nil))))

;;; Result TN generation for record types
;;; Called from src/code/c-call.lisp

;;; Windows verison: large struct uses pointer in RAX, otherwise value in RAX

#+win32
(defun record-result-tn (type state)
  "Handle struct return values."
  (let ((classification (or *cached-struct-classification*
                            (classify-struct type))))
    (setf (result-state-num-results state) 1)
    (if (sb-alien::struct-classification-memory-p classification)
        (make-wired-tn* 'system-area-pointer sap-reg-sc-number rax-offset)
        (make-wired-tn* 'unsigned-byte-64 unsigned-reg-sc-number rax-offset))))

#-win32
(defun record-result-tn (type state)
  "Handle struct return values."
  (let ((classification (or *cached-struct-classification*
                            (classify-struct type))))
    (if (sb-alien::struct-classification-memory-p classification)
        (progn
          (setf (result-state-num-results state) 1)
          (make-wired-tn* 'system-area-pointer sap-reg-sc-number rax-offset))
        (let ((result-tns nil)
              (int-results 0)
              (sse-results 0))
          (dolist (class (sb-alien::struct-classification-register-slots classification))
            (case class
              (:integer
               (push (make-wired-tn* 'unsigned-byte-64
                                     unsigned-reg-sc-number
                                     (result-reg-offset int-results))
                     result-tns)
               (incf int-results))
              (:double
               (push (make-wired-tn* 'double-float
                                     double-reg-sc-number
                                     sse-results)
                     result-tns)
               (incf sse-results))))
          (setf (result-state-num-results state) (+ int-results sse-results))
          (nreverse result-tns)))))

;;; VOPs for struct argument passing
;;; These VOPs load eightbytes from a struct SAP into target registers

(define-vop (load-struct-int-arg)
  (:args (sap :scs (sap-reg)))
  (:info offset)
  (:results (target :scs (unsigned-reg signed-reg)))
  (:generator 5
    (inst mov :qword target (ea offset sap))))

(define-vop (load-struct-sse-arg)
  (:args (sap :scs (sap-reg)))
  (:info offset)
  (:results (target :scs (double-reg single-reg)))
  (:generator 5
    (inst movsd target (ea offset sap))))

;;; VOPs for storing struct result registers to memory
;;; These VOPs store result register values back to memory for struct-by-value returns

(define-vop (store-struct-int-result)
  (:args (value :scs (unsigned-reg signed-reg))
         (sap :scs (sap-reg)))
  (:info offset)
  (:generator 5
    (inst mov :qword (ea offset sap) value)))

(define-vop (store-struct-sse-result)
  (:args (value :scs (double-reg single-reg))
         (sap :scs (sap-reg)))
  (:info offset)
  (:generator 5
    (inst movsd (ea offset sap) value)))

;;; VOP to copy a qword from struct SAP to the C argument stack
;;; Used for passing large structs (>16 bytes) by value
(define-vop (copy-struct-arg-to-stack)
  (:args (sap :scs (sap-reg))
         (nsp :scs (any-reg)))
  (:info src-offset dst-offset)
  (:temporary (:sc unsigned-reg :from (:argument 0)) temp)
  (:generator 5
    (inst mov :qword temp (ea src-offset sap))
    (inst mov :qword (ea dst-offset nsp) temp)))

;;; VOP to move SAP value to integer register (for Windows struct-by-pointer)
(define-vop (load-sap-int-arg)
  (:args (sap :scs (sap-reg)))
  (:results (target :scs (unsigned-reg)))
  (:generator 1
    (move target sap)))

;;; Arg TN generation for record types
;;; Called from src/code/c-call.lisp

;;; Windows: structs >8 bytes passed by pointer, <=8 bytes in
;;; integer register.
#+win32
(defun record-arg-tn (type state)
  "Handle struct arguments."
  (let ((classification (classify-struct type))
        (arg-tn (int-arg state 'unsigned-byte-64
                         unsigned-reg-sc-number unsigned-stack-sc-number)))
    (sb-c::make-arg-tn-loader
     (list arg-tn)
     (if (sb-alien::struct-classification-memory-p classification)
         (lambda (arg call block nsp)
           (declare (ignore nsp))
           (let ((sap-tn (sb-c::lvar-tn call block arg)))
             (sb-c::emit-and-insert-vop
              call block
              (sb-c::template-or-lose 'load-sap-int-arg)
              (sb-c::reference-tn sap-tn nil)
              (sb-c::reference-tn arg-tn t)
              nil
              nil)))
         (lambda (arg call block nsp)
           (declare (ignore nsp))
           (sb-c::emit-and-insert-vop
            call block
            (sb-c::template-or-lose 'load-struct-int-arg)
            (sb-c::reference-tn (sb-c::lvar-tn call block arg) nil)
            (sb-c::reference-tn arg-tn t)
            nil
            (list 0)))))))

;;; System V: structs >16 bytes copied to stack, <=16 bytes in up to 2
;;; registers.
#-win32
(defun record-arg-tn (type state)
  "Handle struct arguments.
   For large structs (>16 bytes), copies to stack per System V AMD64 ABI.
   For small structs, returns a function that emits load VOPs into registers."
  (let ((classification (classify-struct type)))
    (if (sb-alien::struct-classification-memory-p classification)
        ;; Large struct: copy to stack (System V AMD64 ABI)
        ;; The struct is passed by value on the stack, not by pointer
        (let* ((size (sb-alien::struct-classification-size classification))
               (words (ceiling size 8))
               (stack-base (arg-state-stack-frame-size state)))
          ;; Reserve stack slots for the struct
          (incf (arg-state-stack-frame-size state) words)
          ;; Return a function that copies the struct to the stack
          (lambda (arg call block nsp)
            (let ((sap-tn (sb-c::lvar-tn call block arg)))
              (loop for i from 0 below words
                    for src-offset = (* i 8)
                    for dst-offset = (* (+ stack-base i) n-word-bytes)
                    do (sb-c::emit-and-insert-vop
                        call block
                        (sb-c::template-or-lose 'copy-struct-arg-to-stack)
                        (sb-c::reference-tn-list (list sap-tn nsp) nil)
                        nil  ; no results
                        nil  ; insert at end
                        (list src-offset dst-offset))))))
        ;; Small struct: allocate target TNs and return a function to load into them
        (let ((arg-tns nil)
              (offsets nil)
              (offset 0))
          (dolist (class (sb-alien::struct-classification-register-slots classification))
            (case class
              (:integer
               (push (int-arg state 'unsigned-byte-64
                              unsigned-reg-sc-number
                              unsigned-stack-sc-number)
                     arg-tns)
               (push (cons offset :integer) offsets))
              (:double
               (push (float-arg state 'double-float
                                double-reg-sc-number
                                double-stack-sc-number)
                     arg-tns)
               (push (cons offset :double) offsets)))
            (incf offset 8))
          (setf arg-tns (nreverse arg-tns))
          (setf offsets (nreverse offsets))
          ;; Return arg-tn-loader with TNs exposed for register allocator
          (sb-c::make-arg-tn-loader
           arg-tns
           (lambda (arg call block nsp)
             (declare (ignore nsp))
             (let ((sap-tn (sb-c::lvar-tn call block arg)))
               (loop for target-tn in arg-tns
                     for (off . class) in offsets
                     do (let ((vop (ecase class
                                     (:integer 'load-struct-int-arg)
                                     (:double 'load-struct-sse-arg))))
                          (sb-c::emit-and-insert-vop
                           call block
                           (sb-c::template-or-lose vop)
                           (sb-c::reference-tn sap-tn nil)
                           (sb-c::reference-tn target-tn t)
                           nil
                           (list off)))))))))))

;;; VOP to set up hidden struct return pointer in first arg register.
#+win32
(define-vop (set-struct-return-pointer)
  (:args (sap :scs (sap-reg) :target rcx))
  (:temporary (:sc sap-reg :offset rcx-offset) rcx)
  (:generator 1
    (move rcx sap)))

#-win32
(define-vop (set-struct-return-pointer)
  (:args (sap :scs (sap-reg) :target rdi))
  (:temporary (:sc sap-reg :offset rdi-offset :from (:argument 0)) rdi)  ; RDI is the first arg register
  (:generator 1
    (move rdi sap)))

(defun make-call-out-tns (type)
  (let ((arg-state (make-arg-state))
        (result-type (alien-fun-type-result-type type)))
    ;; Check for large struct return and reserve first arg register
    ;; for sret pointer
    ;; Cache the classification to avoid recomputing it in record-result-tn
    (let* ((result-classification
             (when (alien-record-type-p result-type)
               (classify-struct result-type)))
           (large-struct-return-p
             (and result-classification
                  (sb-alien::struct-classification-memory-p result-classification))))
      ;; For large struct returns, consume first int arg register
      ;; so regular arguments start from the second register
      (when large-struct-return-p
        (setf (arg-state-register-args arg-state) 1))
      (collect ((arg-tns))
        (dolist (arg-type (alien-fun-type-arg-types type))
          (arg-tns (invoke-alien-type-method :arg-tn arg-type arg-state)))
        (let ((stack-frame-size (* (arg-state-stack-frame-size arg-state) n-word-bytes))
              ;; Bind cached classification so record-result-tn doesn't recompute
              (*cached-struct-classification* result-classification))
          (values (make-wired-tn* 'positive-fixnum any-reg-sc-number rsp-offset)
                  stack-frame-size
                  (arg-tns)
                  (invoke-alien-type-method :result-tn result-type (make-result-state))
                  ;; 5th value: T if large struct return (sret pointer passed as first arg)
                  large-struct-return-p))))))


(deftransform %alien-funcall ((function type &rest args) * * :node node)
  (aver (sb-c:constant-lvar-p type))
  (let* ((type (sb-c:lvar-value type))
         (env (sb-c::node-lexenv node))
         (arg-types (alien-fun-type-arg-types type))
         (result-type (alien-fun-type-result-type type))
         ;; Large struct returns have a hidden first arg (sret pointer) added by IR1
         #-sb-xc-host
         (large-struct-return-p
           (multiple-value-bind (in-registers-p register-slots size)
               (sb-alien::struct-return-info result-type)
             (declare (ignore register-slots))
             (and size (not in-registers-p))))
         #+sb-xc-host
         (large-struct-return-p nil))
    (aver (= (length arg-types)
             (- (length args) (if large-struct-return-p 1 0))))
    (if (or (some #'(lambda (type)
                      (and (alien-integer-type-p type)
                           (> (sb-alien::alien-integer-type-bits type) 64)))
                  arg-types)
            (and (alien-integer-type-p result-type)
                 (> (sb-alien::alien-integer-type-bits result-type) 64)))
        (collect ((new-args) (lambda-vars) (new-arg-types))
          (dolist (type arg-types)
            (let ((arg (gensym)))
              (lambda-vars arg)
              (cond ((and (alien-integer-type-p type)
                          (> (sb-alien::alien-integer-type-bits type) 64))
                     ;; CLH: FIXME! This should really be
                     ;; #xffffffffffffffff. nyef says: "Passing
                     ;; 128-bit integers to ALIEN functions on x86-64
                     ;; believed to be broken."
                     (new-args `(logand ,arg #xffffffff))
                     (new-args `(ash ,arg -64))
                     (new-arg-types (parse-alien-type '(unsigned 64) env))
                     (if (alien-integer-type-signed type)
                         (new-arg-types (parse-alien-type '(signed 64) env))
                         (new-arg-types (parse-alien-type '(unsigned 64) env))))
                    (t
                     (new-args arg)
                     (new-arg-types type)))))
          (cond ((and (alien-integer-type-p result-type)
                      (> (sb-alien::alien-integer-type-bits result-type) 64))
                 (let ((new-result-type
                        (let ((sb-alien::*values-type-okay* t))
                          (parse-alien-type
                           (if (alien-integer-type-signed result-type)
                               '(values (unsigned 64) (signed 64))
                               '(values (unsigned 64) (unsigned 64)))
                           env))))
                   `(lambda (function type ,@(lambda-vars))
                      (declare (ignore type))
                      (multiple-value-bind (low high)
                          (%alien-funcall function
                                          ',(make-alien-fun-type
                                             :arg-types (new-arg-types)
                                             :result-type new-result-type)
                                          ,@(new-args))
                        (logior low (ash high 64))))))
                (t
                 `(lambda (function type ,@(lambda-vars))
                    (declare (ignore type))
                    (%alien-funcall function
                                    ',(make-alien-fun-type
                                       :arg-types (new-arg-types)
                                       :result-type result-type)
                                    ,@(new-args))))))
        (sb-c::give-up-ir1-transform))))

;;; The ABI is vague about how signed sub-word integer return values
;;; are handled, but since gcc versions >=4.3 no longer do sign
;;; extension in the callee, we need to do it in the caller.  FIXME:
;;; If the value to be extended is known to already be of the target
;;; type at compile time, we can (and should) elide the extension.
(defknown sign-extend ((signed-byte 64) t) fixnum
    (foldable flushable movable))

(defoptimizer (sign-extend derive-type) ((x size))
  (when (sb-c:constant-lvar-p size)
    (specifier-type `(signed-byte ,(sb-c:lvar-value size)))))

(define-vop (sign-extend)
  (:translate sign-extend)
  (:policy :fast-safe)
  (:args (val :scs (signed-reg)))
  (:arg-types signed-num (:constant fixnum))
  (:info size)
  (:results (res :scs (signed-reg)))
  (:result-types fixnum)
  (:generator 1
    (inst movsx `(,(ecase size (8 :byte) (16 :word) (32 :dword)) :qword) res val)))

#-sb-xc-host
(defun sign-extend (x size)
  (declare (type (signed-byte 64) x))
  (ecase size
    (8 (sign-extend x size))
    (16 (sign-extend x size))
    (32 (sign-extend x size))))

;;; There is a troublesome assumption about alien code linkage entries, namely that you
;;; can reference entry + 8 to extract the actual address of the C function.
;;; This is not ideal, for two distinct reasons:
;;;
;;; (1) The linkage entry should contain instructions for GC yieldpoint cooperation,
;;; removing such instructions from call out sites. (You have to inform the GC that
;;; a thread is leaving managed code and entering code that won't execute yieldpoints.)
;;; Clearly this won't work if jumping into the middle of the linkage entry is allowed.
;;;
;;; (2) The CPU has separate I+D caches, and there is a cost to shuttling data between
;;; them. Jumping to an alien linkage entries as they are puts the whole entry into the I
;;; cache (presumably) when the second word should instead be in the D cache.
;;; To optimally structure the entries, all JMPs should precede all data words, like so:
;;;   jmp [RIP+disp]
;;;   jmp [RIP+disp]
;;;   ...
;;;   data ...
;;; And were such change made, it would cease to be valid to jump to an entry + 8.
(define-vop (foreign-symbol-sap)
  (:translate foreign-symbol-sap)
  (:policy :fast-safe)
  (:args)
  (:arg-types (:constant simple-string))
  (:info foreign-symbol)
  (:results (res :scs (sap-reg)))
  (:result-types system-area-pointer)
  (:vop-var vop)
  (:generator 2
    #-immobile-space (inst lea res (ea (make-fixup foreign-symbol :foreign) null-tn))
    #+immobile-space
    (cond ((code-immobile-p vop)
           (inst lea res (rip-relative-ea (make-fixup foreign-symbol :foreign))))
          (t
           (inst mov res (make-fixup foreign-symbol :foreign))
           (inst add res (static-constant-ea alien-linkage-table))))))

(define-vop (foreign-symbol-dataref-sap)
  (:translate foreign-symbol-dataref-sap)
  (:policy :fast-safe)
  (:args)
  (:arg-types (:constant simple-string))
  (:info foreign-symbol)
  (:results (res :scs (sap-reg)))
  (:result-types system-area-pointer)
  (:vop-var vop)
  (:generator 2
    #-immobile-space (inst mov res (ea (make-fixup foreign-symbol :foreign-dataref) null-tn))
    #+immobile-space
    (cond ((code-immobile-p vop)
           (inst mov res (rip-relative-ea (make-fixup foreign-symbol :foreign-dataref))))
          (t
           (inst mov res (static-constant-ea alien-linkage-table))
           (inst mov res (ea (make-fixup foreign-symbol :foreign-dataref) res))))))

#+(or sb-safepoint nonstop-foreign-call)
(defconstant thread-saved-csp-offset -1)

(eval-when (#-sb-xc :compile-toplevel :load-toplevel :execute)
  (defun destroyed-c-registers ()
    ;; Safepoints do not save interrupt contexts to be scanned during
    ;; GCing, it only looks at the stack, so if a register isn't
    ;; spilled it won't be visible to the GC.
    #+(or sb-safepoint nonstop-foreign-call)
    '((:save-p t))
    #-(or sb-safepoint nonstop-foreign-call)
    (let ((gprs (list '#:rcx '#:rdx #-win32 '#:rsi #-win32 '#:rdi
                      '#:r8 '#:r9 '#:r10 '#:r11))
          (vars))
      (append
       (loop for gpr in gprs
             for offset = (symbol-value (intern (concatenate 'string (symbol-name gpr) "-OFFSET") "SB-VM"))
             collect `(:temporary (:sc any-reg :offset ,offset :from :eval :to :result)
                                  ,(car (push gpr vars))))
       (loop for float to 15
             for varname = (format nil "FLOAT~D" float)
             collect `(:temporary (:sc single-reg :offset ,float :from :eval :to :result)
                                  ,(car (push (make-symbol varname) vars))))
       `((:ignore ,@vars))))))

(define-vop (call-out)
  (:args (function :scs (sap-reg)
                   :target rbx)
         (args :more t))
  (:results (results :more t))
  ;; RBX is used to first load the address, allowing the debugger to
  ;; determine which alien was accessed in case it's undefined.
  (:temporary (:sc sap-reg :offset rbx-offset :from (:argument 0)) rbx)
  (:temporary (:sc unsigned-reg :offset rax-offset :to :result) rax)
  #+sb-safepoint
  (:temporary (:sc unsigned-stack :from :eval :to :result) pc-save)
  #+win32
  (:temporary (:sc unsigned-reg :offset r15-offset :from :eval :to :result) r15)
  (:ignore results)
  (:vop-var vop)
  (:generator 0
    (move rbx function)
    (emit-c-call vop rax rbx args
                 sb-alien::*alien-fun-type-varargs-default*
                 #+sb-safepoint pc-save
                 #+win32 rbx))
  #+win32 (:ignore r15)
  . #.(destroyed-c-registers))

;;; Calls to C can generally be made without loading a register
;;; with the function. We receive the function name as an info argument.
(define-vop (call-out-named)
  (:args (args :more t))
  (:results (results :more t))
  (:info c-symbol varargsp)
  (:temporary (:sc unsigned-reg :offset rax-offset :to :result) rax)
  #+sb-safepoint
  (:temporary (:sc unsigned-stack :from :eval :to :result) pc-save)
  #+win32
  (:temporary (:sc unsigned-reg :offset r15-offset :from :eval :to :result) r15)
  #+win32
  (:ignore r15)
  (:temporary (:sc unsigned-reg :offset rbx-offset :from :eval :to :result) rbx)
  (:ignore results)
  (:vop-var vop)
  (:generator 0
    (progn rbx)
    (emit-c-call vop rax c-symbol args varargsp
                 #+sb-safepoint pc-save
                 #+win32 rbx))
  . #.(destroyed-c-registers))

;;; Remember when changing this to check that these work:
;;; - disassembly, undefined alien, and conversion to ELF core
(defun emit-c-call (vop rax fun args varargsp #+sb-safepoint pc-save #+win32 rbx)
  (declare (ignorable varargsp))
  ;; Current PC - don't rely on function to keep it in a form that
  ;; GC understands
  #+sb-safepoint
  (let ((label (gen-label)))
    (inst lea rax (rip-relative-ea label))
    (emit-label label)
    (move pc-save rax))
  (when (sb-c:msan-unpoison sb-c:*compilation*)
    (inst mov rax (thread-slot-ea thread-msan-param-tls-slot))
    ;; Unpoison parameters
    (do ((n 0 (+ n n-word-bytes))
         (arg args (tn-ref-across arg)))
        ((null arg))
      ;; KLUDGE: assume all parameters are 8 bytes or less
      (inst mov :qword (ea n rax) 0)))
  #-win32
  ;; ABI: AL contains amount of arguments passed in XMM registers
  ;; for vararg calls.
  (when varargsp
    (move-immediate rax
                    (loop for tn-ref = args then (tn-ref-across tn-ref)
                          while tn-ref
                          count (eq (sb-name (sc-sb (tn-sc (tn-ref-tn tn-ref))))
                                    'float-registers))))

  ;; Store SP in thread struct, unless the enclosing block says not to

  #+(or sb-safepoint nonstop-foreign-call)
  (when (and #+sb-safepoint
             (policy (sb-c::vop-node vop) (/= sb-c:insert-safepoints 0)))
    (inst mov (thread-slot-ea thread-saved-csp-offset) rsp-tn))

  #+win32 (inst sub rsp-tn #x20)       ;MS_ABI: shadow zone

  ;; Immobile code uses "CALL rel32" to reach the linkage table entry,
  ;; but movable code computes the linkage entry address into RBX first.
  ;; N.B.: if you change how the call is emitted, you will also have to adjust
  ;; the UNDEFINED-ALIEN-TRAMP lisp asm routine to recognize the various shapes
  ;; this instruction sequence can take.
  #-win32
  (pseudo-atomic (:elide-if (not (call-out-pseudo-atomic-p vop)))
    (inst call
          #-immobile-space ; always call via RBX
          (cond ((stringp fun) (inst lea rbx-tn (ea (make-fixup fun :foreign) null-tn)) rbx-tn)
                (t fun))
          #+immobile-space ; sometimes call via RBX
          (if (stringp fun)
              (cond ((code-immobile-p vop) (make-fixup fun :foreign))
                    (t
                     (inst mov rbx-tn (make-fixup fun :foreign))
                     (inst add rbx-tn (static-constant-ea alien-linkage-table))
                     rbx-tn))
              ;; Emit a 3-byte NOP so the undefined-alien routine reads a well-defined byte
              ;; on error. In practice, decoding never seemed to go wrong, but looked fishy
              ;; due to the possibility of any random bytes preceding the call.
              (dolist (b '(#x0f #x1f #x00) fun) (inst byte b)))))

  ;; On win64, calls go through a thunk defined in set_up_win64_seh_data().
  #+win32
  (progn
    (cond ((tn-p fun) (move rbx fun)) ; wasn't this already done by the VOP ?
          ;; Compute address of entrypoint in the alien linkage table into RBX
          ((code-immobile-p vop)
           (inst lea rbx (rip-relative-ea (make-fixup fun :foreign))))
          (t
           #-immobile-space (inst lea rbx (ea (make-fixup fun :foreign) null-tn))
           #+immobile-space
           (progn (inst mov rbx (make-fixup fun :foreign))
                  (inst add rbx (static-constant-ea alien-linkage-table)))))
    (invoke-asm-routine 'call 'seh-trampoline vop))

  ;; For the undefined alien error
  (note-this-location vop :internal-error)
  #+win32 (inst add rsp-tn #x20)       ;MS_ABI: remove shadow space

  ;; Zero the saved CSP, unless this code shouldn't ever stop for GC
  #+sb-safepoint
  (when (policy (sb-c::vop-node vop) (/= sb-c:insert-safepoints 0))
    (inst xor (thread-slot-ea thread-saved-csp-offset) rsp-tn))
  #+nonstop-foreign-call
  (inst mov :qword (thread-slot-ea thread-saved-csp-offset) 0))

(define-vop (alloc-number-stack-space)
  (:info amount)
  (:results (result :scs (sap-reg any-reg)))
  (:result-types system-area-pointer)
  (:generator 0
    (aver (location= result rsp-tn))
    (unless (zerop amount)
      (let ((delta (align-up amount 8)))
        (inst sub rsp-tn delta)))
    ;; C stack must be 16 byte aligned
    (inst and rsp-tn -16)
    (move result rsp-tn)))

(macrolet ((alien-stack-ptr ()
             #+sb-thread `(thread-slot-ea ,(symbol-thread-slot '*alien-stack-pointer*))
             #-sb-thread '(static-symbol-value-ea '*alien-stack-pointer*)))
  (define-vop (alloc-alien-stack-space)
    (:info amount)
    (:results (result :scs (sap-reg any-reg)))
    (:result-types system-area-pointer)
    (:generator 0
      (aver (not (location= result rsp-tn)))
      (unless (zerop amount)
        (let ((delta (align-up amount 8)))
          (inst sub :qword (alien-stack-ptr) delta)))
      (inst mov result (alien-stack-ptr)))))

;;; Callbacks

#-sb-xc-host
(defun alien-callback-accessor-form (type sp offset)
  `(deref (sap-alien (sap+ ,sp ,offset) (* ,type))))

#-sb-xc-host
(defun alien-callback-assembler-wrapper (index result-type argument-types)
  ;; Windows x64 struct-by-value callback rules:
  ;;   1. Struct arguments >8 bytes: caller passes pointer in register
  ;;   2. Struct arguments <=8 bytes: passed in integer register as value
  ;;   3. Struct returns >8 bytes: hidden pointer in RCX (first arg register)
  ;;   4. Struct returns <=8 bytes: returned in RAX
  (labels ((make-tn-maker (sc-name)
             (lambda (offset)
               (make-random-tn (sc-or-lose sc-name) offset)))
           (argument-byte-size (type)
             "Return the number of bytes this argument occupies in the callback vector."
             (ceiling (sb-alien::alien-type-bits type) n-byte-bits))
           (round-up-to-word (bytes)
             (* n-word-bytes (ceiling bytes n-word-bytes))))
    ;; Check for struct return type and classify it
    (let* ((result-classification
             (when (alien-record-type-p result-type)
               (classify-struct result-type)))
           (large-struct-return-p
             (and result-classification
                  (sb-alien::struct-classification-memory-p result-classification)))
           (segment (make-segment))
           (rax rax-tn)
           #+win32 (rcx rcx-tn)
           #-(and win32 sb-thread) (rdi rdi-tn)
           #-(and win32 sb-thread) (rsi rsi-tn)
           (rdx rdx-tn)
           (rbp rbp-tn)
           (rsp rsp-tn)
           #+(and win32 sb-thread) (r8 r8-tn)
           #+win32 (r11 r11-tn)  ; scratch register for struct copy (not an arg register)
           (xmm0 float0-tn)
           #-win32
           (xmm1 float1-tn)
           ([rsp] (ea rsp))
           ;; Calculate total argument vector size in bytes
           (total-arg-bytes
             (loop for type in argument-types
                   sum (round-up-to-word (argument-byte-size type))))
           ;; How many arguments have been copied from the C stack
           (stack-argument-count #-win32 0 #+win32 4)
           ;; Byte offset into argument vector
           (arg-offset 0)
           ;; Count of 8-byte slots consumed (for stack offset calculation)
           (arg-slot-count (ceiling total-arg-bytes n-word-bytes))
           ;; For large struct returns, the hidden pointer is in the first arg register
           ;; (RCX on Windows, RDI on SysV). Skip it in the GPR list.
           ;; On Windows, this also consumes argument slot 0, so skip XMM0 too.
           (gprs (let ((all-gprs (mapcar (make-tn-maker 'any-reg) *c-call-register-arg-offsets*)))
                   (if large-struct-return-p
                       (rest all-gprs)  ; Skip RCX (win32) or RDI (SysV)
                       all-gprs)))
           (fprs (let ((all-fprs (mapcar (make-tn-maker 'double-reg)
                                         ;; Only 8 first XMM registers are used for
                                         ;; passing arguments
                                         (subseq *float-regs* 0 #-win32 8 #+win32 4))))
                   ;; On Windows, when there's a hidden return pointer in RCX (slot 0),
                   ;; the float arguments shift: XMM0 is "consumed" by slot 0, so
                   ;; actual float args start at XMM1.
                   #+win32
                   (if large-struct-return-p
                       (rest all-fprs)
                       all-fprs)
                   #-win32
                   all-fprs))
           ;; Calculate return value slot count (in 8-byte words)
           ;; For large struct returns, we need enough space for the entire struct
           ;; For small structs and primitives, 2 slots (16 bytes) is enough
           (return-slot-count
             (if large-struct-return-p
                 (ceiling (sb-alien::struct-classification-size result-classification) n-word-bytes)
                 2))
           ;; Adjust for alignment (must be even for 16-byte stack alignment)
           (return-slot-count-aligned
             (if (evenp (+ arg-slot-count return-slot-count
                           (if large-struct-return-p
                               1 ;; hidden pointer register saved on the stack
                               0)))
                 return-slot-count
                 (1+ return-slot-count))))
      (assemble (segment 'nil)
        ;; For large struct returns, save the hidden pointer before using it
        ;; Windows: RCX (first arg register), SysV: RDI (first arg register)
        (when large-struct-return-p
          #+win32 (inst push rcx)
          #-win32 (inst push rdi))
        ;; Make room on the stack for argument vector.
        (when (plusp total-arg-bytes)
          (inst sub rsp total-arg-bytes))
        ;; Copy arguments from registers/stack to argument vector
        (dolist (type argument-types)
          (let* ((arg-size (round-up-to-word (argument-byte-size type)))
                 ;; A TN pointing to the stack location where the
                 ;; current argument should be stored for the purposes
                 ;; of ENTER-ALIEN-CALLBACK.
                 (target-tn (ea arg-offset rsp))
                 ;; Offset to C stack args (past return address and our arg vector)
                 (stack-arg-tn (ea (* (+ 1 arg-slot-count stack-argument-count)
                                      n-word-bytes) rsp)))
            (cond
              ;; Struct types
              ((sb-alien::alien-record-type-p type)
               #+win32
               (let* ((classification (classify-struct type))
                      (memory-p (sb-alien::struct-classification-memory-p classification))
                      (struct-size (sb-alien::struct-classification-size classification)))
                 (cond
                   ;; Large struct: pointer passed in register
                   (memory-p
                    (let ((gpr (pop gprs)))
                      (pop fprs) ; Windows: consume paired FPR slot
                      (unless gpr
                        (incf stack-argument-count)
                        (setf gpr rax)
                        (inst mov gpr stack-arg-tn))
                      ;; gpr now contains pointer to struct; copy struct data to arg vector
                      ;; Use r11 as scratch (not an arg register) to avoid clobbering other args
                      (let ((num-words (ceiling struct-size n-word-bytes)))
                        (loop for i from 0 below num-words
                              for dst-off from arg-offset by n-word-bytes
                              do (inst mov r11 (ea (* i n-word-bytes) gpr))
                                 (inst mov (ea dst-off rsp) r11)))))
                   ;; Small struct: single integer register
                   (t
                    (let ((gpr (pop gprs)))
                      (pop fprs)
                      (unless gpr
                        (incf stack-argument-count)
                        (setf gpr rax)
                        (inst mov gpr stack-arg-tn))
                      (inst mov (ea arg-offset rsp) gpr)))))
               #-win32
               (let* ((classification (classify-struct type))
                      (memory-p (sb-alien::struct-classification-memory-p classification))
                      (slots (sb-alien::struct-classification-register-slots classification))
                      (struct-size (sb-alien::struct-classification-size classification)))
                 (cond
                   ;; Large struct (MEMORY class): passed directly on the C stack
                   ;; The caller copies the struct to its stack frame
                   (memory-p
                    (let ((num-words (ceiling struct-size n-word-bytes)))
                      ;; Copy struct data from C stack to our argument vector
                      (loop for i from 0 below num-words
                            for src-off = (* (+ 1 arg-slot-count stack-argument-count i)
                                             n-word-bytes)
                            for dst-off from arg-offset by n-word-bytes
                            do (inst mov rax (ea src-off rsp))
                               (inst mov (ea dst-off rsp) rax))
                      ;; Account for the stack slots consumed
                      (incf stack-argument-count num-words)))
                   ;; Small struct: passed in up to 2 registers per eightbyte
                   (t
                    (loop for class in slots
                          for slot-offset from arg-offset by n-word-bytes
                          do (ecase class
                               (:integer
                                (let ((gpr (pop gprs)))
                                  (unless gpr
                                    (incf stack-argument-count)
                                    (setf gpr rax)
                                    (inst mov gpr (ea (* (+ 1 arg-slot-count stack-argument-count -1)
                                                         n-word-bytes) rsp)))
                                  (inst mov (ea slot-offset rsp) gpr)))
                               (:double
                                (let ((fpr (pop fprs)))
                                  (cond (fpr
                                         (inst movq (ea slot-offset rsp) fpr))
                                        (t
                                         (incf stack-argument-count)
                                         (inst mov rax (ea (* (+ 1 arg-slot-count stack-argument-count -1)
                                                              n-word-bytes) rsp))
                                         (inst mov (ea slot-offset rsp) rax)))))))))))

              ;; Integer/pointer types
              ((not (alien-float-type-p type))
               (let ((gpr (pop gprs)))
                 #+win32 (pop fprs)
                 ;; Argument not in register, copy it from the old
                 ;; stack location to a temporary register.
                 (unless gpr
                   (incf stack-argument-count)
                   (setf gpr rax)
                   (inst mov gpr stack-arg-tn))
                 ;; Copy from either argument register or temporary
                 ;; register to target.
                 (inst mov target-tn gpr)))

              ;; Float types
              ((or (alien-single-float-type-p type)
                   (alien-double-float-type-p type))
               (let ((fpr (pop fprs)))
                 #+win32 (pop gprs)
                 (cond (fpr
                        ;; Copy from float register to target location.
                        (inst movq target-tn fpr))
                       (t
                        ;; Not in float register. Copy from stack to
                        ;; temporary (general purpose) register, and
                        ;; from there to the target location.
                        (incf stack-argument-count)
                        (inst mov rax stack-arg-tn)
                        (inst mov target-tn rax)))))

              (t
               (bug "Unknown alien callback argument type: ~S" type)))
            ;; Advance to next argument slot
            (incf arg-offset arg-size)))

        (macrolet
            ((call-wrapper ()
               ;; Technically this fixup should have an optional arg of
               ;;  (- (ASH SYMBOL-VALUE-SLOT WORD-SHIFT) OTHER-POINTER-LOWTAG)
               ;; but as the fixup is hand-crafted anyway, it doesn't matter.
               `(inst call (rip-relative-ea
                      (make-fixup 'callback-wrapper-trampoline
                                  :immobile-symbol))))) ; arbitraryish flavor
        #-sb-thread
        (progn
          ;; arg0 to ENTER-ALIEN-CALLBACK (trampoline index)
          (inst mov rdx (fixnumize index))
          ;; arg1 to ENTER-ALIEN-CALLBACK (pointer to argument vector)
          (inst mov rdi rsp)
          ;; add room on stack for return value
          (inst sub rsp (* return-slot-count-aligned n-word-bytes))
          ;; arg2 to ENTER-ALIEN-CALLBACK (pointer to return value)
          (inst mov rsi rsp)

          ;; Make new frame
          (inst push rbp)
          (inst mov  rbp rsp)

          ;; Call
          (call-wrapper)

          ;; Back! Restore frame
          (inst leave))

        #+sb-thread
        (progn
          ;; arg0 to ENTER-ALIEN-CALLBACK (trampoline index)
          (inst mov #-win32 rdi #+win32 rcx (fixnumize index))
          ;; arg1 to ENTER-ALIEN-CALLBACK (pointer to argument vector)
          (inst mov #-win32 rsi #+win32 rdx rsp)
          ;; add room on stack for return value
          (inst sub rsp (* return-slot-count-aligned n-word-bytes))
          ;; arg2 to ENTER-ALIEN-CALLBACK (pointer to return value)
          (inst mov #-win32 rdx #+win32 r8 rsp)
          ;; Make new frame
          (inst push rbp)
          (inst mov  rbp rsp)
          #+win32 (inst sub rsp #x20)
          #+win32 (inst and rsp #x-20)
          ;; Call
          (call-wrapper)

          ;; Back! Restore frame
          (inst leave)))

        ;; Result now on top of stack, put it in the right register
        (cond
          ((or (alien-integer-type-p result-type)
               (alien-pointer-type-p result-type)
               (alien-type-= #.(parse-alien-type 'system-area-pointer nil)
                             result-type))
           (inst mov rax [rsp]))
          ((or (alien-single-float-type-p result-type)
               (alien-double-float-type-p result-type))
           (inst movq xmm0 [rsp]))
          ((alien-void-type-p result-type))
          ;; Struct return types
          ((alien-record-type-p result-type)
           #+win32
           ;; Windows: large structs via hidden pointer (from RCX), small structs in RAX
           (cond
             ;; Large struct: copy result to hidden pointer location, return pointer
             (large-struct-return-p
              (let ((struct-size (sb-alien::struct-classification-size result-classification)))
                ;; Retrieve saved hidden pointer (was pushed at start from RCX)
                (inst mov rax (ea (* (+ arg-slot-count return-slot-count-aligned) n-word-bytes) rsp))
                ;; Copy struct data from stack to hidden pointer destination
                (loop for off from 0 below struct-size by 8
                      do (inst mov rdx (ea off rsp))
                         (inst mov (ea off rax) rdx))))
             ;; Small struct (<=8 bytes): just load into RAX
             (t
              (inst mov rax [rsp])))
           #-win32
           ;; SysV: large structs via hidden pointer (from RDI), small structs in RAX/RDX/XMM0/XMM1
           (cond
             (large-struct-return-p
              (let ((struct-size (sb-alien::struct-classification-size result-classification)))
                (inst mov rax (ea (* (+ arg-slot-count return-slot-count-aligned) n-word-bytes) rsp))
                (loop for off from 0 below struct-size by 8
                      do (inst mov rdx (ea off rsp))
                         (inst mov (ea off rax) rdx))))
             ;; Small struct: copy to registers based on classification
             (t
              (let ((slots (sb-alien::struct-classification-register-slots result-classification))
                    (int-reg-idx 0)
                    (sse-reg-idx 0))
                (loop for slot in slots
                      for offset from 0 by 8
                      do (ecase slot
                           (:integer
                            (let ((target (case int-reg-idx
                                            (0 rax)
                                            (1 rdx))))
                              (inst mov target (ea offset rsp)))
                            (incf int-reg-idx))
                           (:double
                            (let ((target (case sse-reg-idx
                                            (0 xmm0)
                                            (1 xmm1))))
                              (inst movq target (ea offset rsp)))
                            (incf sse-reg-idx))))))))
          (t
           (error "Unrecognized alien type: ~A" result-type)))

        ;; Pop the arguments and the return value from the stack to get
        ;; the return address at top of stack.

        (inst add rsp (* (+ arg-slot-count return-slot-count-aligned
                            (if large-struct-return-p
                                1
                                0))
                         n-word-bytes))
        ;; Return
        (inst ret))
      (finalize-segment segment)
      ;; Now that the segment is done, convert it to a static
      ;; vector we can point foreign code to.
      (let* ((buffer (sb-assem:segment-buffer segment))
             (result (make-static-vector (length buffer)
                                         :element-type '(unsigned-byte 8)
                                         :initial-contents buffer)))
        ;; This is an ad-hoc substitute for the general fixup logic, due to
        ;; absence of a code component. Even the machine-dependent part is not
        ;; useful since it wants to call CODE-INSTRUCTIONS.
        (let* ((notes (sb-assem::segment-fixup-notes segment))
               (note (car notes)))
          (when note
            (aver (eq (fixup-note-kind note) :rel32))
            ;; +4 is because RIP-relative EA is relative to following instruction
            (let* ((pc (sap+ (vector-sap result) (+ (fixup-note-position note) 4)))
                   (fixup (fixup-note-fixup note))
                   (ea (+ nil-value (ea-disp (static-symbol-value-ea (fixup-name fixup)))))
                   (disp (sap- (int-sap ea) pc)))
              (setf (signed-sap-ref-32  (vector-sap result) (fixup-note-position note))
                    disp))))
        result))))
