/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.util;

import java.util.Objects;
import java.util.function.Consumer;
import org.jboss.logging.Logger;
import org.keycloak.common.ClientConnection;
import org.keycloak.common.Profile;
import org.keycloak.models.AuthenticatedClientSessionModel;
import org.keycloak.models.ClientModel;
import org.keycloak.models.ImpersonationSessionNote;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.UserSessionProvider;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.models.utils.UserSessionModelDelegate;
import org.keycloak.protocol.oidc.TokenManager;
import org.keycloak.protocol.oidc.encode.AccessTokenContext;
import org.keycloak.protocol.oidc.encode.TokenContextEncoderProvider;
import org.keycloak.representations.AccessToken;
import org.keycloak.representations.RefreshToken;
import org.keycloak.services.Urls;
import org.keycloak.services.managers.AuthenticationManager;
import org.keycloak.services.managers.UserSessionManager;
import org.keycloak.services.util.DefaultClientSessionContext;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.sessions.RootAuthenticationSessionModel;

public class UserSessionUtil {
    private static final Logger logger = Logger.getLogger(UserSessionUtil.class);

    public static UserSessionValidationResult findValidSessionForIdentityCookie(KeycloakSession session, RealmModel realm, AccessToken token, Consumer<UserSessionModel> invalidSessionCallback) {
        return UserSessionUtil.findValidSession(session, realm, token, null, AccessTokenContext.SessionType.ONLINE, false, true, invalidSessionCallback);
    }

    public static UserSessionValidationResult findValidSessionForRefreshToken(KeycloakSession session, RealmModel realm, RefreshToken token, ClientModel client, Consumer<UserSessionModel> invalidSessionCallback) {
        AccessTokenContext.SessionType sessionType;
        if ("Offline".equals(token.getType())) {
            sessionType = AccessTokenContext.SessionType.OFFLINE;
        } else if ("Refresh".equals(token.getType())) {
            sessionType = AccessTokenContext.SessionType.ONLINE;
        } else {
            return UserSessionValidationResult.error("invalid_token_type");
        }
        return UserSessionUtil.findValidSession(session, realm, (AccessToken)token, client, sessionType, Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.TOKEN_EXCHANGE), false, invalidSessionCallback);
    }

    public static UserSessionValidationResult findValidSessionForAccessToken(KeycloakSession session, RealmModel realm, AccessToken token, ClientModel client, Consumer<UserSessionModel> invalidSessionCallback) {
        AccessTokenContext accessTokenContext = ((TokenContextEncoderProvider)session.getProvider(TokenContextEncoderProvider.class)).getTokenContextFromTokenId(token.getId());
        AccessTokenContext.SessionType sessionType = accessTokenContext.getSessionType();
        return UserSessionUtil.findValidSession(session, realm, token, client, sessionType, Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.TOKEN_EXCHANGE), false, invalidSessionCallback);
    }

    private static UserSessionValidationResult findValidSession(KeycloakSession session, RealmModel realm, AccessToken token, ClientModel client, AccessTokenContext.SessionType sessionType, boolean allowImpersonationFallback, boolean skipCheckClient, Consumer<UserSessionModel> invalidSessionCallback) {
        logger.tracef("Lookup user session with the sessionType '%s'. Token session id: %s", (Object)sessionType, (Object)token.getSessionId());
        if (token.getSessionId() == null) {
            if (sessionType.isAllowTransientUserSession()) {
                return UserSessionUtil.createTransientSessionForClient(session, realm, token, client);
            }
            return UserSessionValidationResult.error("user_session_not_found");
        }
        UserSessionProvider userSessionProvider = session.sessions();
        UserSessionModel userSession = null;
        if (sessionType.isAllowLookupOnlineUserSession()) {
            AuthenticatedClientSessionModel clientSession = null;
            if (skipCheckClient || sessionType.isAllowTransientClientSession()) {
                userSession = userSessionProvider.getUserSession(realm, token.getSessionId());
            } else {
                userSession = userSessionProvider.getUserSessionIfClientExists(realm, token.getSessionId(), false, client.getId());
                if (userSession != null && !UserSessionUtil.checkTokenIssuedAt(token, clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId()))) {
                    return UserSessionValidationResult.error("invalid_token", userSession, invalidSessionCallback);
                }
                if (userSession == null && allowImpersonationFallback) {
                    userSession = UserSessionUtil.getUserSessionWithImpersonatorClient(session, realm, token.getSessionId(), false, client.getId());
                }
            }
            if (AuthenticationManager.isSessionValid(realm, userSession)) {
                if (!UserSessionUtil.checkTokenIssuedAt(token, userSession)) {
                    return UserSessionValidationResult.error("invalid_token", userSession, invalidSessionCallback);
                }
                if (sessionType.isAllowTransientClientSession()) {
                    userSession = UserSessionUtil.createTransientSessionForClient(session, userSession, client);
                    return UserSessionValidationResult.validSession(session, userSession);
                }
                return UserSessionValidationResult.validSession(session, userSession);
            }
        }
        UserSessionModel offlineUserSession = null;
        if (sessionType.isAllowLookupOfflineUserSession()) {
            AuthenticatedClientSessionModel offlineClientSession = null;
            if (sessionType.isAllowTransientClientSession()) {
                offlineUserSession = userSessionProvider.getOfflineUserSession(realm, token.getSessionId());
            } else {
                offlineUserSession = userSessionProvider.getUserSessionIfClientExists(realm, token.getSessionId(), true, client.getId());
                if (offlineUserSession != null && !UserSessionUtil.checkTokenIssuedAt(token, offlineClientSession = offlineUserSession.getAuthenticatedClientSessionByClient(client.getId()))) {
                    return UserSessionValidationResult.error("invalid_token", offlineUserSession, invalidSessionCallback);
                }
            }
            if (AuthenticationManager.isSessionValid(realm, offlineUserSession)) {
                if (!UserSessionUtil.checkTokenIssuedAt(token, offlineUserSession)) {
                    return UserSessionValidationResult.error("invalid_token", offlineUserSession, invalidSessionCallback);
                }
                if (sessionType.isAllowTransientClientSession()) {
                    offlineUserSession = UserSessionUtil.createTransientSessionForClient(session, offlineUserSession, client);
                    return UserSessionValidationResult.validSession(session, offlineUserSession);
                }
                return UserSessionValidationResult.validSession(session, offlineUserSession);
            }
        }
        if (userSession == null && offlineUserSession == null) {
            logger.debugf("User session '%s' not found or doesn't have client attached on it", (Object)token.getSessionId());
            return UserSessionValidationResult.error("user_session_not_found");
        }
        logger.debugf("Session '%s' expired", (Object)token.getSessionId());
        return UserSessionValidationResult.error("session_expired", userSession != null ? userSession : offlineUserSession, invalidSessionCallback);
    }

    public static UserSessionModel createTransientUserSession(KeycloakSession session, final UserSessionModel userSession) {
        if (userSession.getPersistenceState() == UserSessionModel.SessionPersistenceState.TRANSIENT) {
            throw new IllegalArgumentException("Not expected to invoke this method with the transient session");
        }
        UserSessionModel transientSession = new UserSessionManager(session).createUserSession(userSession.getId(), userSession.getRealm(), userSession.getUser(), userSession.getLoginUsername(), userSession.getIpAddress(), userSession.getAuthMethod(), userSession.isRememberMe(), userSession.getBrokerSessionId(), userSession.getBrokerUserId(), UserSessionModel.SessionPersistenceState.TRANSIENT);
        userSession.getNotes().entrySet().forEach(e -> transientSession.setNote((String)e.getKey(), (String)e.getValue()));
        String noteValue = userSession.isOffline() ? "offline" : "online";
        transientSession.setNote("created_from_persistent", noteValue);
        return new UserSessionModelDelegate(transientSession){

            public int getStarted() {
                return userSession.getStarted();
            }
        };
    }

    public static boolean isOfflineAccessGranted(KeycloakSession session, AuthenticatedClientSessionModel clientSession) {
        if (clientSession == null) {
            return false;
        }
        DefaultClientSessionContext clientSessionCtx = DefaultClientSessionContext.fromClientSessionAndScopeParameter(clientSession, "offline_access", session);
        return clientSessionCtx.getClientScopesStream().anyMatch(s -> "offline_access".equals(s.getName()));
    }

    private static void attachAuthenticationSession(KeycloakSession session, UserSessionModel userSession, ClientModel client) {
        RootAuthenticationSessionModel rootAuthSession = session.authenticationSessions().createRootAuthenticationSession(userSession.getRealm());
        AuthenticationSessionModel authSession = rootAuthSession.createAuthenticationSession(client);
        authSession.setAuthenticatedUser(userSession.getUser());
        authSession.setProtocol("openid-connect");
        authSession.setClientNote("iss", Urls.realmIssuer(session.getContext().getUri().getBaseUri(), userSession.getRealm().getName()));
        AuthenticationManager.setClientScopesInSession(session, authSession);
        TokenManager.attachAuthenticationSession(session, userSession, authSession);
    }

    private static UserSessionModel createTransientSessionForClient(KeycloakSession session, UserSessionModel userSession, ClientModel client) {
        UserSessionModel transientSession = UserSessionUtil.createTransientUserSession(session, userSession);
        UserSessionUtil.attachAuthenticationSession(session, transientSession, client);
        return transientSession;
    }

    private static UserSessionValidationResult createTransientSessionForClient(KeycloakSession session, RealmModel realm, AccessToken token, ClientModel client) {
        UserModel user = TokenManager.lookupUserFromStatelessToken(session, realm, token);
        if (user == null) {
            logger.debug((Object)"Transient User not found");
            return UserSessionValidationResult.error("user_not_found");
        }
        if (!user.isEnabled()) {
            logger.debugf("User '%s' disabled", (Object)user.getUsername());
            return UserSessionValidationResult.error("user_disabled");
        }
        ClientConnection clientConnection = session.getContext().getConnection();
        UserSessionModel userSession = new UserSessionManager(session).createUserSession(KeycloakModelUtils.generateId(), realm, user, user.getUsername(), clientConnection.getRemoteHost(), "client_auth", false, null, null, UserSessionModel.SessionPersistenceState.TRANSIENT);
        UserSessionUtil.attachAuthenticationSession(session, userSession, client);
        return UserSessionValidationResult.validSession(session, userSession);
    }

    private static boolean checkTokenIssuedAt(AccessToken token, UserSessionModel userSession) {
        if (token.isIssuedBeforeSessionStart((long)userSession.getStarted())) {
            logger.debug((Object)"Stale token for user session");
            return false;
        }
        return true;
    }

    private static boolean checkTokenIssuedAt(AccessToken token, AuthenticatedClientSessionModel clientSession) {
        if (token.isIssuedBeforeSessionStart((long)clientSession.getStarted())) {
            logger.debug((Object)"Stale token for client session");
            return false;
        }
        return true;
    }

    public static UserSessionModel getUserSessionWithImpersonatorClient(KeycloakSession session, RealmModel realm, String userSessionId, boolean offline, String clientUUID) {
        return session.sessions().getUserSessionWithPredicate(realm, userSessionId, offline, userSession -> Objects.equals(clientUUID, userSession.getNote(ImpersonationSessionNote.IMPERSONATOR_CLIENT.toString())));
    }

    public static class UserSessionValidationResult {
        private final UserSessionModel userSession;
        private final String error;

        private static UserSessionValidationResult validSession(KeycloakSession session, UserSessionModel userSession) {
            session.getContext().setUserSession(userSession);
            return new UserSessionValidationResult(userSession, null);
        }

        private static UserSessionValidationResult error(String error) {
            return new UserSessionValidationResult(null, error);
        }

        private static UserSessionValidationResult error(String error, UserSessionModel invalidUserSession, Consumer<UserSessionModel> invalidSessionCallback) {
            invalidSessionCallback.accept(invalidUserSession);
            return new UserSessionValidationResult(null, error);
        }

        private UserSessionValidationResult(UserSessionModel userSession, String error) {
            this.userSession = userSession;
            this.error = error;
        }

        public UserSessionModel getUserSession() {
            return this.userSession;
        }

        public String getError() {
            return this.error;
        }
    }
}

