/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.IntMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.UnsignedLongs;
import com.google.errorprone.annotations.InlineMe;
import java.math.RoundingMode;

@GwtCompatible
public final class LongMath {
    @VisibleForTesting
    static final long MAX_SIGNED_POWER_OF_TWO = 0x4000000000000000L;
    @VisibleForTesting
    static final long MAX_POWER_OF_SQRT2_UNSIGNED = -5402926248376769404L;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{19, 18, 18, 18, 18, 17, 17, 17, 16, 16, 16, 15, 15, 15, 15, 14, 14, 14, 13, 13, 13, 12, 12, 12, 12, 11, 11, 11, 10, 10, 10, 9, 9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0};
    @GwtIncompatible
    @VisibleForTesting
    static final long[] powersOf10 = new long[]{1L, 10L, 100L, 1000L, 10000L, 100000L, 1000000L, 10000000L, 100000000L, 1000000000L, 10000000000L, 100000000000L, 1000000000000L, 10000000000000L, 100000000000000L, 1000000000000000L, 10000000000000000L, 100000000000000000L, 1000000000000000000L};
    @GwtIncompatible
    @VisibleForTesting
    static final long[] halfPowersOf10 = new long[]{3L, 31L, 316L, 3162L, 31622L, 316227L, 3162277L, 31622776L, 316227766L, 3162277660L, 31622776601L, 316227766016L, 3162277660168L, 31622776601683L, 316227766016837L, 3162277660168379L, 31622776601683793L, 316227766016837933L, 3162277660168379331L};
    @VisibleForTesting
    static final long FLOOR_SQRT_MAX_LONG = 3037000499L;
    static final long[] factorials = new long[]{1L, 1L, 2L, 6L, 24L, 120L, 720L, 5040L, 40320L, 362880L, 3628800L, 39916800L, 479001600L, 6227020800L, 87178291200L, 1307674368000L, 20922789888000L, 355687428096000L, 6402373705728000L, 121645100408832000L, 2432902008176640000L};
    static final int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 3810779, 121977, 16175, 4337, 1733, 887, 534, 361, 265, 206, 169, 143, 125, 111, 101, 94, 88, 83, 79, 76, 74, 72, 70, 69, 68, 67, 67, 66, 66, 66, 66};
    @VisibleForTesting
    static final int[] biggestSimpleBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 2642246, 86251, 11724, 3218, 1313, 684, 419, 287, 214, 169, 139, 119, 105, 95, 87, 81, 76, 73, 70, 68, 66, 64, 63, 62, 62, 61, 61, 61};
    private static final int SIEVE_30 = -545925251;
    private static final long[][] millerRabinBaseSets = new long[][]{{291830L, 126401071349994536L}, {885594168L, 725270293939359937L, 3569819667048198375L}, {273919523040L, 15L, 7363882082L, 992620450144556L}, {47636622961200L, 2L, 2570940L, 211991001L, 3749873356L}, {7999252175582850L, 2L, 4130806001517L, 149795463772692060L, 186635894390467037L, 3967304179347715805L}, {585226005592931976L, 2L, 123635709730000L, 9233062284813009L, 43835965440333360L, 761179012939631437L, 1263739024124850375L}, {Long.MAX_VALUE, 2L, 325L, 9375L, 28178L, 450775L, 9780504L, 1795265022L}};

    public static long ceilingPowerOfTwo(long x15) {
        MathPreconditions.checkPositive("x", x15);
        if (x15 > 0x4000000000000000L) {
            throw new ArithmeticException("ceilingPowerOfTwo(" + x15 + ") is not representable as a long");
        }
        return 1L << -Long.numberOfLeadingZeros(x15 - 1L);
    }

    public static long floorPowerOfTwo(long x15) {
        MathPreconditions.checkPositive("x", x15);
        return 1L << 63 - Long.numberOfLeadingZeros(x15);
    }

    public static boolean isPowerOfTwo(long x15) {
        return x15 > 0L & (x15 & x15 - 1L) == 0L;
    }

    @VisibleForTesting
    static int lessThanBranchFree(long x15, long y15) {
        return (int)((x15 - y15 ^ 0xFFFFFFFFFFFFFFFFL ^ 0xFFFFFFFFFFFFFFFFL) >>> 63);
    }

    public static int log2(long x15, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x15);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(LongMath.isPowerOfTwo(x15));
            }
            case DOWN: 
            case FLOOR: {
                return 63 - Long.numberOfLeadingZeros(x15);
            }
            case UP: 
            case CEILING: {
                return 64 - Long.numberOfLeadingZeros(x15 - 1L);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int leadingZeros = Long.numberOfLeadingZeros(x15);
                long cmp = -5402926248376769404L >>> leadingZeros;
                int logFloor = 63 - leadingZeros;
                return logFloor + LongMath.lessThanBranchFree(cmp, x15);
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    @GwtIncompatible
    public static int log10(long x15, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x15);
        int logFloor = LongMath.log10Floor(x15);
        long floorPow = powersOf10[logFloor];
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(x15 == floorPow);
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return logFloor + LongMath.lessThanBranchFree(floorPow, x15);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return logFloor + LongMath.lessThanBranchFree(halfPowersOf10[logFloor], x15);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    static int log10Floor(long x15) {
        byte y15 = maxLog10ForLeadingZeros[Long.numberOfLeadingZeros(x15)];
        return y15 - LongMath.lessThanBranchFree(x15, powersOf10[y15]);
    }

    @GwtIncompatible
    public static long pow(long b15, int k15) {
        MathPreconditions.checkNonNegative("exponent", k15);
        if (-2L <= b15 && b15 <= 2L) {
            switch ((int)b15) {
                case 0: {
                    return k15 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k15 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    return k15 < 64 ? 1L << k15 : 0L;
                }
                case -2: {
                    if (k15 < 64) {
                        return (k15 & 1) == 0 ? 1L << k15 : -(1L << k15);
                    }
                    return 0L;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        while (true) {
            switch (k15) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return accum * b15;
                }
            }
            accum *= (k15 & 1) == 0 ? 1L : b15;
            b15 *= b15;
            k15 >>= 1;
        }
    }

    @GwtIncompatible
    public static long sqrt(long x15, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x15);
        if (LongMath.fitsInInt(x15)) {
            return IntMath.sqrt((int)x15, mode);
        }
        long guess = (long)Math.sqrt(x15);
        long guessSquared = guess * guess;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(guessSquared == x15);
                return guess;
            }
            case DOWN: 
            case FLOOR: {
                if (x15 < guessSquared) {
                    return guess - 1L;
                }
                return guess;
            }
            case UP: 
            case CEILING: {
                if (x15 > guessSquared) {
                    return guess + 1L;
                }
                return guess;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long sqrtFloor = guess - (long)(x15 < guessSquared ? 1 : 0);
                long halfSquare = sqrtFloor * sqrtFloor + sqrtFloor;
                return sqrtFloor + (long)LongMath.lessThanBranchFree(halfSquare, x15);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static long divide(long p15, long q15, RoundingMode mode) {
        boolean increment;
        Preconditions.checkNotNull(mode);
        long div = p15 / q15;
        long rem = p15 - q15 * div;
        if (rem == 0L) {
            return div;
        }
        int signum = 1 | (int)((p15 ^ q15) >> 63);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(rem == 0L);
            }
            case DOWN: {
                increment = false;
                break;
            }
            case UP: {
                increment = true;
                break;
            }
            case CEILING: {
                increment = signum > 0;
                break;
            }
            case FLOOR: {
                increment = signum < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long absRem = Math.abs(rem);
                long cmpRemToHalfDivisor = absRem - (Math.abs(q15) - absRem);
                if (cmpRemToHalfDivisor == 0L) {
                    increment = mode == RoundingMode.HALF_UP || mode == RoundingMode.HALF_EVEN && (div & 1L) != 0L;
                    break;
                }
                increment = cmpRemToHalfDivisor > 0L;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? div + (long)signum : div;
    }

    @GwtIncompatible
    public static int mod(long x15, int m15) {
        return (int)LongMath.mod(x15, (long)m15);
    }

    @GwtIncompatible
    public static long mod(long x15, long m15) {
        if (m15 <= 0L) {
            throw new ArithmeticException("Modulus must be positive");
        }
        return Math.floorMod(x15, m15);
    }

    public static long gcd(long a15, long b15) {
        MathPreconditions.checkNonNegative("a", a15);
        MathPreconditions.checkNonNegative("b", b15);
        if (a15 == 0L) {
            return b15;
        }
        if (b15 == 0L) {
            return a15;
        }
        int aTwos = Long.numberOfTrailingZeros(a15);
        a15 >>= aTwos;
        int bTwos = Long.numberOfTrailingZeros(b15);
        b15 >>= bTwos;
        while (a15 != b15) {
            long delta = a15 - b15;
            long minDeltaOrZero = delta & delta >> 63;
            a15 = delta - minDeltaOrZero - minDeltaOrZero;
            b15 += minDeltaOrZero;
            a15 >>= Long.numberOfTrailingZeros(a15);
        }
        return a15 << Math.min(aTwos, bTwos);
    }

    @InlineMe(replacement="Math.addExact(a, b)")
    public static long checkedAdd(long a15, long b15) {
        return Math.addExact(a15, b15);
    }

    @InlineMe(replacement="Math.subtractExact(a, b)")
    public static long checkedSubtract(long a15, long b15) {
        return Math.subtractExact(a15, b15);
    }

    @InlineMe(replacement="Math.multiplyExact(a, b)")
    public static long checkedMultiply(long a15, long b15) {
        return Math.multiplyExact(a15, b15);
    }

    @GwtIncompatible
    public static long checkedPow(long b15, int k15) {
        MathPreconditions.checkNonNegative("exponent", k15);
        if (b15 >= -2L & b15 <= 2L) {
            switch ((int)b15) {
                case 0: {
                    return k15 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k15 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    MathPreconditions.checkNoOverflow(k15 < 63, "checkedPow", b15, (long)k15);
                    return 1L << k15;
                }
                case -2: {
                    MathPreconditions.checkNoOverflow(k15 < 64, "checkedPow", b15, (long)k15);
                    return (k15 & 1) == 0 ? 1L << k15 : -1L << k15;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        while (true) {
            switch (k15) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return Math.multiplyExact(accum, b15);
                }
            }
            if ((k15 & 1) != 0) {
                accum = Math.multiplyExact(accum, b15);
            }
            if ((k15 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-3037000499L <= b15 && b15 <= 3037000499L, "checkedPow", b15, (long)k15);
            b15 *= b15;
        }
    }

    public static long saturatedAdd(long a15, long b15) {
        long naiveSum;
        if ((a15 ^ b15) < 0L | (a15 ^ (naiveSum = a15 + b15)) >= 0L) {
            return naiveSum;
        }
        return Long.MAX_VALUE + (naiveSum >>> 63 ^ 1L);
    }

    public static long saturatedSubtract(long a15, long b15) {
        long naiveDifference;
        if ((a15 ^ b15) >= 0L | (a15 ^ (naiveDifference = a15 - b15)) >= 0L) {
            return naiveDifference;
        }
        return Long.MAX_VALUE + (naiveDifference >>> 63 ^ 1L);
    }

    public static long saturatedMultiply(long a15, long b15) {
        int leadingZeros = Long.numberOfLeadingZeros(a15) + Long.numberOfLeadingZeros(a15 ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(b15) + Long.numberOfLeadingZeros(b15 ^ 0xFFFFFFFFFFFFFFFFL);
        if (leadingZeros > 65) {
            return a15 * b15;
        }
        long limit = Long.MAX_VALUE + ((a15 ^ b15) >>> 63);
        if (leadingZeros < 64 | a15 < 0L & b15 == Long.MIN_VALUE) {
            return limit;
        }
        long result = a15 * b15;
        if (a15 == 0L || result / a15 == b15) {
            return result;
        }
        return limit;
    }

    public static long saturatedPow(long b15, int k15) {
        MathPreconditions.checkNonNegative("exponent", k15);
        if (b15 >= -2L & b15 <= 2L) {
            switch ((int)b15) {
                case 0: {
                    return k15 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k15 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    if (k15 >= 63) {
                        return Long.MAX_VALUE;
                    }
                    return 1L << k15;
                }
                case -2: {
                    if (k15 >= 64) {
                        return Long.MAX_VALUE + (long)(k15 & 1);
                    }
                    return (k15 & 1) == 0 ? 1L << k15 : -1L << k15;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        long limit = Long.MAX_VALUE + (b15 >>> 63 & (long)(k15 & 1));
        while (true) {
            switch (k15) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return LongMath.saturatedMultiply(accum, b15);
                }
            }
            if ((k15 & 1) != 0) {
                accum = LongMath.saturatedMultiply(accum, b15);
            }
            if ((k15 >>= 1) <= 0) continue;
            if (-3037000499L > b15 | b15 > 3037000499L) {
                return limit;
            }
            b15 *= b15;
        }
    }

    @GwtIncompatible
    public static long factorial(int n15) {
        MathPreconditions.checkNonNegative("n", n15);
        return n15 < factorials.length ? factorials[n15] : Long.MAX_VALUE;
    }

    public static long binomial(int n15, int k15) {
        MathPreconditions.checkNonNegative("n", n15);
        MathPreconditions.checkNonNegative("k", k15);
        Preconditions.checkArgument(k15 <= n15, "k (%s) > n (%s)", k15, n15);
        if (k15 > n15 >> 1) {
            k15 = n15 - k15;
        }
        switch (k15) {
            case 0: {
                return 1L;
            }
            case 1: {
                return n15;
            }
        }
        if (n15 < factorials.length) {
            return factorials[n15] / (factorials[k15] * factorials[n15 - k15]);
        }
        if (k15 >= biggestBinomials.length || n15 > biggestBinomials[k15]) {
            return Long.MAX_VALUE;
        }
        if (k15 < biggestSimpleBinomials.length && n15 <= biggestSimpleBinomials[k15]) {
            long result = n15--;
            for (int i15 = 2; i15 <= k15; ++i15) {
                result *= (long)n15;
                result /= (long)i15;
                --n15;
            }
            return result;
        }
        int nBits = LongMath.log2(n15, RoundingMode.CEILING);
        long result = 1L;
        long numerator = n15--;
        long denominator = 1L;
        int numeratorBits = nBits;
        int i16 = 2;
        while (i16 <= k15) {
            if (numeratorBits + nBits < 63) {
                numerator *= (long)n15;
                denominator *= (long)i16;
                numeratorBits += nBits;
            } else {
                result = LongMath.multiplyFraction(result, numerator, denominator);
                numerator = n15;
                denominator = i16;
                numeratorBits = nBits;
            }
            ++i16;
            --n15;
        }
        return LongMath.multiplyFraction(result, numerator, denominator);
    }

    static long multiplyFraction(long x15, long numerator, long denominator) {
        if (x15 == 1L) {
            return numerator / denominator;
        }
        long commonDivisor = LongMath.gcd(x15, denominator);
        return (x15 /= commonDivisor) * (numerator / (denominator /= commonDivisor));
    }

    static boolean fitsInInt(long x15) {
        return (long)((int)x15) == x15;
    }

    public static long mean(long x15, long y15) {
        return (x15 & y15) + ((x15 ^ y15) >> 1);
    }

    @GwtIncompatible
    public static boolean isPrime(long n15) {
        if (n15 < 2L) {
            MathPreconditions.checkNonNegative("n", n15);
            return false;
        }
        if (n15 < 66L) {
            long mask = 722865708377213483L;
            return (mask >> (int)n15 - 2 & 1L) != 0L;
        }
        if ((0xDF75D77D & 1 << (int)(n15 % 30L)) != 0) {
            return false;
        }
        if (n15 % 7L == 0L || n15 % 11L == 0L || n15 % 13L == 0L) {
            return false;
        }
        if (n15 < 289L) {
            return true;
        }
        for (long[] baseSet : millerRabinBaseSets) {
            if (n15 > baseSet[0]) continue;
            for (int i15 = 1; i15 < baseSet.length; ++i15) {
                if (MillerRabinTester.test(baseSet[i15], n15)) continue;
                return false;
            }
            return true;
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static double roundToDouble(long x15, RoundingMode mode) {
        double roundArbitrarily = x15;
        long roundArbitrarilyAsLong = (long)roundArbitrarily;
        int cmpXToRoundArbitrarily = roundArbitrarilyAsLong == Long.MAX_VALUE ? -1 : Long.compare(x15, roundArbitrarilyAsLong);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(cmpXToRoundArbitrarily == 0);
                return roundArbitrarily;
            }
            case FLOOR: {
                return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
            }
            case CEILING: {
                return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
            }
            case DOWN: {
                if (x15 >= 0L) {
                    return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
                }
                return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
            }
            case UP: {
                if (x15 >= 0L) {
                    return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
                }
                return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int diff;
                long roundCeiling;
                double roundCeilingAsDouble;
                long roundFloor;
                double roundFloorAsDouble;
                if (cmpXToRoundArbitrarily >= 0) {
                    roundFloorAsDouble = roundArbitrarily;
                    roundFloor = roundArbitrarilyAsLong;
                    roundCeilingAsDouble = Math.nextUp(roundArbitrarily);
                    roundCeiling = (long)Math.ceil(roundCeilingAsDouble);
                } else {
                    roundCeilingAsDouble = roundArbitrarily;
                    roundCeiling = roundArbitrarilyAsLong;
                    roundFloorAsDouble = DoubleUtils.nextDown(roundArbitrarily);
                    roundFloor = (long)Math.floor(roundFloorAsDouble);
                }
                long deltaToFloor = x15 - roundFloor;
                long deltaToCeiling = roundCeiling - x15;
                if (roundCeiling == Long.MAX_VALUE) {
                    ++deltaToCeiling;
                }
                if ((diff = Long.compare(deltaToFloor, deltaToCeiling)) < 0) {
                    return roundFloorAsDouble;
                }
                if (diff > 0) {
                    return roundCeilingAsDouble;
                }
                switch (mode) {
                    case HALF_EVEN: {
                        return (DoubleUtils.getSignificand(roundFloorAsDouble) & 1L) == 0L ? roundFloorAsDouble : roundCeilingAsDouble;
                    }
                    case HALF_DOWN: {
                        return x15 >= 0L ? roundFloorAsDouble : roundCeilingAsDouble;
                    }
                    case HALF_UP: {
                        return x15 >= 0L ? roundCeilingAsDouble : roundFloorAsDouble;
                    }
                }
                throw new AssertionError((Object)"impossible");
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    public static long saturatedAbs(long x15) {
        return x15 == Long.MIN_VALUE ? Long.MAX_VALUE : Math.abs(x15);
    }

    private LongMath() {
    }

    private static enum MillerRabinTester {
        SMALL{

            @Override
            long mulMod(long a15, long b15, long m15) {
                return a15 * b15 % m15;
            }

            @Override
            long squareMod(long a15, long m15) {
                return a15 * a15 % m15;
            }
        }
        ,
        LARGE{

            private long plusMod(long a15, long b15, long m15) {
                return a15 >= m15 - b15 ? a15 + b15 - m15 : a15 + b15;
            }

            private long times2ToThe32Mod(long a15, long m15) {
                int shift;
                int remainingPowersOf2 = 32;
                do {
                    shift = Math.min(remainingPowersOf2, Long.numberOfLeadingZeros(a15));
                    a15 = UnsignedLongs.remainder(a15 << shift, m15);
                } while ((remainingPowersOf2 -= shift) > 0);
                return a15;
            }

            @Override
            long mulMod(long a15, long b15, long m15) {
                long aHi = a15 >>> 32;
                long bHi = b15 >>> 32;
                long aLo = a15 & 0xFFFFFFFFL;
                long bLo = b15 & 0xFFFFFFFFL;
                long result = this.times2ToThe32Mod(aHi * bHi, m15);
                if ((result += aHi * bLo) < 0L) {
                    result = UnsignedLongs.remainder(result, m15);
                }
                result += aLo * bHi;
                result = this.times2ToThe32Mod(result, m15);
                return this.plusMod(result, UnsignedLongs.remainder(aLo * bLo, m15), m15);
            }

            @Override
            long squareMod(long a15, long m15) {
                long aHi = a15 >>> 32;
                long aLo = a15 & 0xFFFFFFFFL;
                long result = this.times2ToThe32Mod(aHi * aHi, m15);
                long hiLo = aHi * aLo * 2L;
                if (hiLo < 0L) {
                    hiLo = UnsignedLongs.remainder(hiLo, m15);
                }
                result += hiLo;
                result = this.times2ToThe32Mod(result, m15);
                return this.plusMod(result, UnsignedLongs.remainder(aLo * aLo, m15), m15);
            }
        };


        static boolean test(long base, long n15) {
            return (n15 <= 3037000499L ? SMALL : LARGE).testWitness(base, n15);
        }

        abstract long mulMod(long var1, long var3, long var5);

        abstract long squareMod(long var1, long var3);

        private long powMod(long a15, long p15, long m15) {
            long res = 1L;
            while (p15 != 0L) {
                if ((p15 & 1L) != 0L) {
                    res = this.mulMod(res, a15, m15);
                }
                a15 = this.squareMod(a15, m15);
                p15 >>= 1;
            }
            return res;
        }

        private boolean testWitness(long base, long n15) {
            int r15 = Long.numberOfTrailingZeros(n15 - 1L);
            long d15 = n15 - 1L >> r15;
            if ((base %= n15) == 0L) {
                return true;
            }
            long a15 = this.powMod(base, d15, n15);
            if (a15 == 1L) {
                return true;
            }
            int j15 = 0;
            while (a15 != n15 - 1L) {
                if (++j15 == r15) {
                    return false;
                }
                a15 = this.squareMod(a15, n15);
            }
            return true;
        }
    }
}

