/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import com.google.common.math.DoubleUtils;
import com.google.common.math.IntMath;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.math.ToDoubleRounder;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

@GwtCompatible
public final class BigIntegerMath {
    @VisibleForTesting
    static final int SQRT2_PRECOMPUTE_THRESHOLD = 256;
    @VisibleForTesting
    static final BigInteger SQRT2_PRECOMPUTED_BITS = new BigInteger("16a09e667f3bcc908b2fb1366ea957d3e3adec17512775099da2f590b0667322a", 16);
    private static final double LN_10 = Math.log(10.0);
    private static final double LN_2 = Math.log(2.0);

    public static BigInteger ceilingPowerOfTwo(BigInteger x15) {
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(x15, RoundingMode.CEILING));
    }

    public static BigInteger floorPowerOfTwo(BigInteger x15) {
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(x15, RoundingMode.FLOOR));
    }

    public static boolean isPowerOfTwo(BigInteger x15) {
        Preconditions.checkNotNull(x15);
        return x15.signum() > 0 && x15.getLowestSetBit() == x15.bitLength() - 1;
    }

    public static int log2(BigInteger x15, RoundingMode mode) {
        MathPreconditions.checkPositive("x", Preconditions.checkNotNull(x15));
        int logFloor = x15.bitLength() - 1;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(BigIntegerMath.isPowerOfTwo(x15));
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return BigIntegerMath.isPowerOfTwo(x15) ? logFloor : logFloor + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                if (logFloor < 256) {
                    BigInteger halfPower = SQRT2_PRECOMPUTED_BITS.shiftRight(256 - logFloor);
                    if (x15.compareTo(halfPower) <= 0) {
                        return logFloor;
                    }
                    return logFloor + 1;
                }
                BigInteger x25 = x15.pow(2);
                int logX2Floor = x25.bitLength() - 1;
                return logX2Floor < 2 * logFloor + 1 ? logFloor : logFloor + 1;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int log10(BigInteger x15, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x15);
        if (BigIntegerMath.fitsInLong(x15)) {
            return LongMath.log10(x15.longValue(), mode);
        }
        int approxLog10 = (int)((double)BigIntegerMath.log2(x15, RoundingMode.FLOOR) * LN_2 / LN_10);
        BigInteger approxPow = BigInteger.TEN.pow(approxLog10);
        int approxCmp = approxPow.compareTo(x15);
        if (approxCmp > 0) {
            do {
                --approxLog10;
            } while ((approxCmp = (approxPow = approxPow.divide(BigInteger.TEN)).compareTo(x15)) > 0);
        } else {
            BigInteger nextPow = BigInteger.TEN.multiply(approxPow);
            int nextCmp = nextPow.compareTo(x15);
            while (nextCmp <= 0) {
                ++approxLog10;
                approxPow = nextPow;
                approxCmp = nextCmp;
                nextPow = BigInteger.TEN.multiply(approxPow);
                nextCmp = nextPow.compareTo(x15);
            }
        }
        int floorLog = approxLog10;
        BigInteger floorPow = approxPow;
        int floorCmp = approxCmp;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(floorCmp == 0);
            }
            case DOWN: 
            case FLOOR: {
                return floorLog;
            }
            case UP: 
            case CEILING: {
                return floorPow.equals(x15) ? floorLog : floorLog + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                BigInteger x25 = x15.pow(2);
                BigInteger halfPowerSquared = floorPow.pow(2).multiply(BigInteger.TEN);
                return x25.compareTo(halfPowerSquared) <= 0 ? floorLog : floorLog + 1;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static BigInteger sqrt(BigInteger x15, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x15);
        if (BigIntegerMath.fitsInLong(x15)) {
            return BigInteger.valueOf(LongMath.sqrt(x15.longValue(), mode));
        }
        BigInteger sqrtFloor = BigIntegerMath.sqrtFloor(x15);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(sqrtFloor.pow(2).equals(x15));
            }
            case DOWN: 
            case FLOOR: {
                return sqrtFloor;
            }
            case UP: 
            case CEILING: {
                int sqrtFloorInt = sqrtFloor.intValue();
                boolean sqrtFloorIsExact = sqrtFloorInt * sqrtFloorInt == x15.intValue() && sqrtFloor.pow(2).equals(x15);
                return sqrtFloorIsExact ? sqrtFloor : sqrtFloor.add(BigInteger.ONE);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                BigInteger halfSquare = sqrtFloor.pow(2).add(sqrtFloor);
                return halfSquare.compareTo(x15) >= 0 ? sqrtFloor : sqrtFloor.add(BigInteger.ONE);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    private static BigInteger sqrtFloor(BigInteger x15) {
        BigInteger sqrt0;
        int log2 = BigIntegerMath.log2(x15, RoundingMode.FLOOR);
        if (log2 < 1023) {
            sqrt0 = BigIntegerMath.sqrtApproxWithDoubles(x15);
        } else {
            int shift = log2 - 52 & 0xFFFFFFFE;
            sqrt0 = BigIntegerMath.sqrtApproxWithDoubles(x15.shiftRight(shift)).shiftLeft(shift >> 1);
        }
        BigInteger sqrt1 = sqrt0.add(x15.divide(sqrt0)).shiftRight(1);
        if (sqrt0.equals(sqrt1)) {
            return sqrt0;
        }
        while ((sqrt1 = (sqrt0 = sqrt1).add(x15.divide(sqrt0)).shiftRight(1)).compareTo(sqrt0) < 0) {
        }
        return sqrt0;
    }

    @GwtIncompatible
    private static BigInteger sqrtApproxWithDoubles(BigInteger x15) {
        return DoubleMath.roundToBigInteger(Math.sqrt(DoubleUtils.bigToDouble(x15)), RoundingMode.HALF_EVEN);
    }

    @GwtIncompatible
    public static double roundToDouble(BigInteger x15, RoundingMode mode) {
        return BigIntegerToDoubleRounder.INSTANCE.roundToDouble(x15, mode);
    }

    @GwtIncompatible
    public static BigInteger divide(BigInteger p15, BigInteger q15, RoundingMode mode) {
        BigDecimal pDec = new BigDecimal(p15);
        BigDecimal qDec = new BigDecimal(q15);
        return pDec.divide(qDec, 0, mode).toBigIntegerExact();
    }

    public static BigInteger factorial(int n15) {
        MathPreconditions.checkNonNegative("n", n15);
        if (n15 < LongMath.factorials.length) {
            return BigInteger.valueOf(LongMath.factorials[n15]);
        }
        int approxSize = IntMath.divide(n15 * IntMath.log2(n15, RoundingMode.CEILING), 64, RoundingMode.CEILING);
        ArrayList<BigInteger> bignums = new ArrayList<BigInteger>(approxSize);
        int startingNumber = LongMath.factorials.length;
        long product = LongMath.factorials[startingNumber - 1];
        int shift = Long.numberOfTrailingZeros(product);
        int productBits = LongMath.log2(product >>= shift, RoundingMode.FLOOR) + 1;
        int bits = LongMath.log2(startingNumber, RoundingMode.FLOOR) + 1;
        int nextPowerOfTwo = 1 << bits - 1;
        for (long num = (long)startingNumber; num <= (long)n15; ++num) {
            if ((num & (long)nextPowerOfTwo) != 0L) {
                nextPowerOfTwo <<= 1;
                ++bits;
            }
            int tz4 = Long.numberOfTrailingZeros(num);
            long normalizedNum = num >> tz4;
            shift += tz4;
            int normalizedBits = bits - tz4;
            if (normalizedBits + productBits >= 64) {
                bignums.add(BigInteger.valueOf(product));
                product = 1L;
                productBits = 0;
            }
            productBits = LongMath.log2(product *= normalizedNum, RoundingMode.FLOOR) + 1;
        }
        if (product > 1L) {
            bignums.add(BigInteger.valueOf(product));
        }
        return BigIntegerMath.listProduct(bignums).shiftLeft(shift);
    }

    static BigInteger listProduct(List<BigInteger> nums) {
        return BigIntegerMath.listProduct(nums, 0, nums.size());
    }

    static BigInteger listProduct(List<BigInteger> nums, int start, int end) {
        switch (end - start) {
            case 0: {
                return BigInteger.ONE;
            }
            case 1: {
                return nums.get(start);
            }
            case 2: {
                return nums.get(start).multiply(nums.get(start + 1));
            }
            case 3: {
                return nums.get(start).multiply(nums.get(start + 1)).multiply(nums.get(start + 2));
            }
        }
        int m15 = end + start >>> 1;
        return BigIntegerMath.listProduct(nums, start, m15).multiply(BigIntegerMath.listProduct(nums, m15, end));
    }

    public static BigInteger binomial(int n15, int k15) {
        int bits;
        MathPreconditions.checkNonNegative("n", n15);
        MathPreconditions.checkNonNegative("k", k15);
        Preconditions.checkArgument(k15 <= n15, "k (%s) > n (%s)", k15, n15);
        if (k15 > n15 >> 1) {
            k15 = n15 - k15;
        }
        if (k15 < LongMath.biggestBinomials.length && n15 <= LongMath.biggestBinomials[k15]) {
            return BigInteger.valueOf(LongMath.binomial(n15, k15));
        }
        BigInteger accum = BigInteger.ONE;
        long numeratorAccum = n15;
        long denominatorAccum = 1L;
        int numeratorBits = bits = LongMath.log2(n15, RoundingMode.CEILING);
        for (int i15 = 1; i15 < k15; ++i15) {
            int p15 = n15 - i15;
            int q15 = i15 + 1;
            if (numeratorBits + bits >= 63) {
                accum = accum.multiply(BigInteger.valueOf(numeratorAccum)).divide(BigInteger.valueOf(denominatorAccum));
                numeratorAccum = p15;
                denominatorAccum = q15;
                numeratorBits = bits;
                continue;
            }
            numeratorAccum *= (long)p15;
            denominatorAccum *= (long)q15;
            numeratorBits += bits;
        }
        return accum.multiply(BigInteger.valueOf(numeratorAccum)).divide(BigInteger.valueOf(denominatorAccum));
    }

    @GwtIncompatible
    static boolean fitsInLong(BigInteger x15) {
        return x15.bitLength() <= 63;
    }

    private BigIntegerMath() {
    }

    @GwtIncompatible
    private static final class BigIntegerToDoubleRounder
    extends ToDoubleRounder<BigInteger> {
        static final BigIntegerToDoubleRounder INSTANCE = new BigIntegerToDoubleRounder();

        private BigIntegerToDoubleRounder() {
        }

        @Override
        double roundToDoubleArbitrarily(BigInteger bigInteger) {
            return DoubleUtils.bigToDouble(bigInteger);
        }

        @Override
        int sign(BigInteger bigInteger) {
            return bigInteger.signum();
        }

        @Override
        BigInteger toX(double d15, RoundingMode mode) {
            return DoubleMath.roundToBigInteger(d15, mode);
        }

        @Override
        BigInteger minus(BigInteger a15, BigInteger b15) {
            return a15.subtract(b15);
        }
    }
}

