/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.string.translate.libretranslate;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import docking.widgets.SelectFromListDialog;
import ghidra.app.plugin.core.string.translate.libretranslate.LibreTranslatePlugin;
import ghidra.app.services.StringTranslationService;
import ghidra.docking.settings.Settings;
import ghidra.net.HttpClients;
import ghidra.program.model.data.DataUtilities;
import ghidra.program.model.data.StringDataInstance;
import ghidra.program.model.data.TranslationSettingsDefinition;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Program;
import ghidra.program.util.ProgramLocation;
import ghidra.util.HelpLocation;
import ghidra.util.Msg;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.CancelledListener;
import ghidra.util.task.TaskLauncher;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpTimeoutException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

public class LibreTranslateStringTranslationService
implements StringTranslationService {
    static final String CONTENT_TYPE_JSON = "application/json";
    static final String CONTENT_TYPE_HEADER = "Content-Type";
    private static final String GHIDRA_USER_AGENT = "Ghidra";
    private URI serverURI;
    private String apiKey;
    private LibreTranslatePlugin.SOURCE_LANGUAGE_OPTION sourceLanguageOption;
    private String targetLanguageCode;
    private List<SupportedLanguage> supportedLanguages;
    private int batchSize;
    private int maxRetryCount = 3;
    private int httpTimeout;
    private int httpTimeoutPerString;

    public LibreTranslateStringTranslationService(URI serverURI, String apiKey, LibreTranslatePlugin.SOURCE_LANGUAGE_OPTION sourceLanguageOption, String targetLanguageCode, int batchSize, int httpTimeout, int httpTimeoutPerString) {
        String path = serverURI.getPath();
        this.serverURI = path.endsWith("/") ? serverURI : serverURI.resolve(path + "/");
        this.apiKey = Objects.requireNonNullElse(apiKey, "");
        this.sourceLanguageOption = sourceLanguageOption;
        this.targetLanguageCode = targetLanguageCode;
        this.batchSize = Math.clamp((long)batchSize, 1, batchSize);
        this.httpTimeout = Math.clamp((long)httpTimeout, 1, httpTimeout);
        this.httpTimeoutPerString = Math.clamp((long)httpTimeoutPerString, 1, httpTimeoutPerString);
    }

    @Override
    public String getTranslationServiceName() {
        return "LibreTranslate";
    }

    @Override
    public HelpLocation getHelpLocation() {
        return new HelpLocation("LibreTranslatePlugin", "LibreTranslatePlugin");
    }

    @Override
    public void translate(Program program, List<ProgramLocation> stringLocations, StringTranslationService.TranslateOptions options) {
        Msg.info((Object)this, (Object)"LibreTranslate translate %d strings using %s".formatted(stringLocations.size(), this.serverURI));
        TaskLauncher.launchModal((String)"Translate Strings", monitor -> {
            String langCode;
            monitor.initialize((long)stringLocations.size(), "Gathering strings");
            List<Map.Entry<ProgramLocation, String>> sourceStrings = stringLocations.stream().map(progLoc -> Map.entry(progLoc, DataUtilities.getDataAtLocation((ProgramLocation)progLoc))).map(entry -> {
                monitor.incrementProgress();
                StringDataInstance sdi = StringDataInstance.getStringDataInstance((Data)((Data)entry.getValue()));
                String s = sdi.getStringValue();
                return s != null ? Map.entry((ProgramLocation)entry.getKey(), s) : null;
            }).filter(Objects::nonNull).toList();
            switch (this.sourceLanguageOption) {
                case PROMPT: {
                    try {
                        this.ensureSupportedLanguages(monitor);
                        SupportedLanguage selectedLang = (SupportedLanguage)SelectFromListDialog.selectFromList(this.supportedLanguages, (String)"Choose Source Language", (String)"Choose", SupportedLanguage::getDescription);
                        if (selectedLang == null) {
                            return;
                        }
                        langCode = selectedLang.langCode;
                        break;
                    }
                    catch (IOException e) {
                        this.showError("Error Fetching Supported Languages", "Failed to retrieve list of supported languages", e);
                        return;
                    }
                }
                default: {
                    langCode = "auto";
                }
            }
            try {
                List<ProgramLocation> successfulStrings = this.translate(program, sourceStrings, langCode, monitor);
                int failCount = sourceStrings.size() - successfulStrings.size();
                if (failCount > 0) {
                    Msg.showWarn((Object)this, null, (String)"Translation Incomplete", (Object)"%d of %d strings not translated".formatted(failCount, stringLocations.size()));
                }
            }
            catch (IOException e) {
                this.showError("LibreTranslate Error", "Error when translating strings", e);
            }
        });
    }

    private List<ProgramLocation> translate(Program program, List<Map.Entry<ProgramLocation, String>> srcStrings, String srcLangCode, TaskMonitor monitor) throws IOException {
        ArrayList<ProgramLocation> successfulStrings = new ArrayList<ProgramLocation>();
        program.withTransaction("Translate strings", () -> {
            try {
                monitor.initialize((long)srcStrings.size(), "Translating strings");
                for (int srcIndex = 0; srcIndex < srcStrings.size(); srcIndex += this.batchSize) {
                    monitor.checkCancelled();
                    List subList = srcStrings.subList(srcIndex, Math.min(srcStrings.size(), srcIndex + this.batchSize));
                    List<String> subListStrs = subList.stream().map(e -> (String)e.getValue()).toList();
                    long start_ts = System.currentTimeMillis();
                    String jsonResponseStr = null;
                    for (int retryCount = 0; retryCount < this.maxRetryCount; ++retryCount) {
                        long timeout = (this.httpTimeout + subListStrs.size() * this.httpTimeoutPerString) * (retryCount + 1);
                        HttpRequest request = this.createTranslateRequest(subListStrs, srcLangCode, timeout);
                        if (retryCount != 0) {
                            monitor.setMessage("Retrying translate request (%d)".formatted(retryCount));
                        }
                        try {
                            jsonResponseStr = this.asyncRequest(request, HttpResponse.BodyHandlers.ofString(), monitor);
                            break;
                        }
                        catch (HttpTimeoutException e2) {
                            if (retryCount == this.maxRetryCount - 1) {
                                throw new IOException("Timeout during translate request, %d of %d strings completed".formatted(successfulStrings.size(), srcStrings.size()), e2);
                            }
                            Msg.error((Object)this, (Object)"LibreTranslate timeout on translate request for: %s".formatted(subListStrs));
                            continue;
                        }
                    }
                    List<String> subResults = this.parseTranslateResponse(jsonResponseStr, subListStrs);
                    for (int resultIndex = 0; resultIndex < subResults.size(); ++resultIndex) {
                        ProgramLocation progLoc = (ProgramLocation)((Map.Entry)subList.get(resultIndex)).getKey();
                        String xlatedValue = subResults.get(resultIndex);
                        if (xlatedValue == null || xlatedValue.trim().isEmpty()) continue;
                        Data data = DataUtilities.getDataAtLocation((ProgramLocation)progLoc);
                        TranslationSettingsDefinition.TRANSLATION.setTranslatedValue(data, xlatedValue);
                        TranslationSettingsDefinition.TRANSLATION.setShowTranslated((Settings)data, true);
                        monitor.increment();
                        successfulStrings.add(progLoc);
                    }
                    long elapsed = System.currentTimeMillis() - start_ts;
                    int sps = subList.size() / Math.max(1, (int)(elapsed / 1000L));
                    Msg.debug((Object)this, (Object)"LibreTranslate translate batch %d/%d strings, %dms".formatted(successfulStrings.size(), srcStrings.size(), elapsed));
                    monitor.setMessage("Translating strings (%d strings per second)".formatted(sps));
                }
            }
            catch (CancelledException cancelledException) {
                // empty catch block
            }
        });
        Msg.info((Object)this, (Object)"Finished LibreTranslate, %d/%d strings".formatted(srcStrings.size(), successfulStrings.size()));
        return successfulStrings;
    }

    private HttpRequest createTranslateRequest(List<String> sourceStrings, String sourceLangCode, long timeout) throws IOException {
        Map<String, Object> requestParams = Map.of("q", sourceStrings, "source", sourceLangCode, "target", this.targetLanguageCode, "format", "text", "alternatives", 0, "api_key", this.apiKey);
        HttpRequest request = this.request("translate", timeout).header(CONTENT_TYPE_HEADER, CONTENT_TYPE_JSON).POST(LibreTranslateStringTranslationService.ofJsonEncodedParams(requestParams)).build();
        return request;
    }

    private List<String> parseTranslateResponse(String jsonResponseStr, List<String> requestedStrs) throws IOException {
        ArrayList<String> results = new ArrayList<String>();
        try {
            JsonObject json = JsonParser.parseString((String)jsonResponseStr).getAsJsonObject();
            JsonElement xlatedTextEle = json.get("translatedText");
            if (xlatedTextEle == null) {
                throw new JsonSyntaxException("Bad json data for translatedText value");
            }
            JsonArray xlatedTexts = xlatedTextEle.getAsJsonArray();
            if (xlatedTexts.size() != requestedStrs.size()) {
                throw new IllegalStateException("LibreTranslate response size mismatch");
            }
            for (int resultIndex = 0; resultIndex < xlatedTexts.size(); ++resultIndex) {
                results.add(xlatedTexts.get(resultIndex).getAsString());
            }
            return results;
        }
        catch (JsonSyntaxException | IllegalStateException e) {
            throw new IOException("Bad data in json response", e);
        }
        finally {
            Msg.error((Object)this, (Object)("Error parsing translate result: " + this.resultToSafeStr(jsonResponseStr)));
        }
    }

    private void ensureSupportedLanguages(TaskMonitor monitor) throws IOException {
        try {
            if (this.supportedLanguages == null || this.supportedLanguages.isEmpty()) {
                this.supportedLanguages = this.getSupportedLanguages(monitor);
                if (this.supportedLanguages != null && !this.supportedLanguages.isEmpty()) {
                    this.supportedLanguages.add(0, new SupportedLanguage("Autodetect", "auto", List.of()));
                }
            }
        }
        catch (CancelledException e) {
            throw new IOException("Failed to get supported language list: request cancelled");
        }
    }

    public List<SupportedLanguage> getSupportedLanguages(TaskMonitor monitor) throws IOException, CancelledException {
        HttpRequest request = this.request("languages").header(CONTENT_TYPE_HEADER, CONTENT_TYPE_JSON).GET().build();
        String jsonResponseStr = this.asyncRequest(request, HttpResponse.BodyHandlers.ofString(), monitor);
        try {
            JsonArray json = JsonParser.parseString((String)jsonResponseStr).getAsJsonArray();
            ArrayList<SupportedLanguage> results = new ArrayList<SupportedLanguage>();
            for (int i = 0; i < json.size(); ++i) {
                SupportedLanguage supportedLang = this.parseSupportedLangJson(json.get(i).getAsJsonObject());
                results.add(supportedLang);
            }
            Collections.sort(results, (o1, o2) -> o1.langCode.compareTo(o2.langCode));
            return results;
        }
        catch (JsonSyntaxException | IllegalStateException e) {
            throw new IOException("Bad data in json response: " + jsonResponseStr, e);
        }
    }

    private SupportedLanguage parseSupportedLangJson(JsonObject obj) {
        if (obj.get("code") == null || obj.get("name") == null || obj.get("targets") == null) {
            throw new JsonSyntaxException("Bad json data for supported language");
        }
        String srcLangCode = obj.get("code").getAsString();
        String srcLangName = obj.get("name").getAsString();
        JsonArray targets = obj.get("targets").getAsJsonArray();
        ArrayList<String> targetLangCodes = new ArrayList<String>();
        targets.forEach(targetEle -> targetLangCodes.add(targetEle.getAsString()));
        return new SupportedLanguage(srcLangName, srcLangCode, targetLangCodes);
    }

    private <T> T asyncRequest(HttpRequest request, HttpResponse.BodyHandler<T> bodyHandler, TaskMonitor monitor) throws CancelledException, IOException {
        CompletableFuture futureResponse = HttpClients.getHttpClient().sendAsync(request, bodyHandler);
        CancelledListener l = () -> futureResponse.cancel(true);
        monitor.addCancelledListener(l);
        try {
            HttpResponse<T> response = futureResponse.get();
            int statusCode = response.statusCode();
            if (statusCode != 200) {
                Msg.debug((Object)this, (Object)"HTTP request [%s], response: %d, body: %s".formatted(request.uri(), statusCode, this.resultToSafeStr(response.body().toString())));
                throw new IOException("Bad HTTP result [%d] for request [%s]".formatted(statusCode, request.uri()));
            }
            String responseContentType = response.headers().firstValue(CONTENT_TYPE_HEADER).orElse("missing");
            if (!CONTENT_TYPE_JSON.equals(responseContentType)) {
                throw new IOException("Bad content-type in result: [%s]".formatted(responseContentType));
            }
            T t = response.body();
            return t;
        }
        catch (InterruptedException e) {
            throw new CancelledException("Request canceled");
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            Msg.error((Object)this, (Object)"Error during HTTP request [%s]".formatted(request.uri()), (Throwable)cause);
            throw cause instanceof IOException ? (IOException)cause : new IOException("Error during HTTP request", cause);
        }
        finally {
            monitor.removeCancelledListener(l);
        }
    }

    private static HttpRequest.BodyPublisher ofJsonEncodedParams(Map<String, Object> params) {
        JsonObject obj = new JsonObject();
        params.forEach((k, v) -> {
            if (v instanceof String) {
                String str = (String)v;
                obj.addProperty(k, str);
            } else if (v instanceof Number) {
                Number num = (Number)v;
                obj.addProperty(k, num);
            } else if (v instanceof Boolean) {
                Boolean bool = (Boolean)v;
                obj.addProperty(k, bool);
            } else if (v instanceof List) {
                List list = (List)v;
                JsonArray jsonArray = new JsonArray();
                for (Object listEle : list) {
                    jsonArray.add(listEle.toString());
                }
                obj.add(k, (JsonElement)jsonArray);
            }
        });
        return HttpRequest.BodyPublishers.ofString(obj.toString());
    }

    private HttpRequest.Builder request(String str) throws IOException {
        return this.request(str, this.httpTimeout);
    }

    private HttpRequest.Builder request(String str, long timeoutMS) throws IOException {
        try {
            return HttpRequest.newBuilder(this.serverURI.resolve(str)).timeout(Duration.ofMillis(timeoutMS)).setHeader("User-Agent", GHIDRA_USER_AGENT);
        }
        catch (IllegalArgumentException e) {
            throw new IOException(e);
        }
    }

    private void showError(String title, String msg, Throwable th) {
        String summary = th.getMessage();
        if (summary == null || summary.isBlank()) {
            summary = th.getClass().getSimpleName();
        }
        Msg.showError((Object)this, null, (String)title, (Object)"%s: %s\n\nLibreTranslate server URL: %s".formatted(msg, summary, this.serverURI), (Throwable)th);
    }

    private String resultToSafeStr(String s) {
        if (((String)s).length() > 200) {
            s = ((String)s).substring(0, 200) + "....";
        }
        return s;
    }

    public record SupportedLanguage(String name, String langCode, List<String> targets) {
        public String getDescription() {
            return "%s (%s)".formatted(this.name, this.langCode);
        }
    }
}

