﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/resource-groups/ResourceGroupsRequest.h>
#include <aws/resource-groups/ResourceGroups_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ResourceGroups {
namespace Model {

/**
 */
class UntagRequest : public ResourceGroupsRequest {
 public:
  AWS_RESOURCEGROUPS_API UntagRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "Untag"; }

  AWS_RESOURCEGROUPS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon resource name (ARN) of the resource group from which to remove
   * tags. The command removed both the specified keys and any values associated with
   * those keys.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  UntagRequest& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The keys of the tags to be removed.</p>
   */
  inline const Aws::Vector<Aws::String>& GetKeys() const { return m_keys; }
  inline bool KeysHasBeenSet() const { return m_keysHasBeenSet; }
  template <typename KeysT = Aws::Vector<Aws::String>>
  void SetKeys(KeysT&& value) {
    m_keysHasBeenSet = true;
    m_keys = std::forward<KeysT>(value);
  }
  template <typename KeysT = Aws::Vector<Aws::String>>
  UntagRequest& WithKeys(KeysT&& value) {
    SetKeys(std::forward<KeysT>(value));
    return *this;
  }
  template <typename KeysT = Aws::String>
  UntagRequest& AddKeys(KeysT&& value) {
    m_keysHasBeenSet = true;
    m_keys.emplace_back(std::forward<KeysT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::Vector<Aws::String> m_keys;
  bool m_keysHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResourceGroups
}  // namespace Aws
