﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/athena/AthenaRequest.h>
#include <aws/athena/Athena_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Athena {
namespace Model {

/**
 */
class GetTableMetadataRequest : public AthenaRequest {
 public:
  AWS_ATHENA_API GetTableMetadataRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetTableMetadata"; }

  AWS_ATHENA_API Aws::String SerializePayload() const override;

  AWS_ATHENA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the data catalog that contains the database and table metadata to
   * return.</p>
   */
  inline const Aws::String& GetCatalogName() const { return m_catalogName; }
  inline bool CatalogNameHasBeenSet() const { return m_catalogNameHasBeenSet; }
  template <typename CatalogNameT = Aws::String>
  void SetCatalogName(CatalogNameT&& value) {
    m_catalogNameHasBeenSet = true;
    m_catalogName = std::forward<CatalogNameT>(value);
  }
  template <typename CatalogNameT = Aws::String>
  GetTableMetadataRequest& WithCatalogName(CatalogNameT&& value) {
    SetCatalogName(std::forward<CatalogNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database that contains the table metadata to return.</p>
   */
  inline const Aws::String& GetDatabaseName() const { return m_databaseName; }
  inline bool DatabaseNameHasBeenSet() const { return m_databaseNameHasBeenSet; }
  template <typename DatabaseNameT = Aws::String>
  void SetDatabaseName(DatabaseNameT&& value) {
    m_databaseNameHasBeenSet = true;
    m_databaseName = std::forward<DatabaseNameT>(value);
  }
  template <typename DatabaseNameT = Aws::String>
  GetTableMetadataRequest& WithDatabaseName(DatabaseNameT&& value) {
    SetDatabaseName(std::forward<DatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the table for which metadata is returned.</p>
   */
  inline const Aws::String& GetTableName() const { return m_tableName; }
  inline bool TableNameHasBeenSet() const { return m_tableNameHasBeenSet; }
  template <typename TableNameT = Aws::String>
  void SetTableName(TableNameT&& value) {
    m_tableNameHasBeenSet = true;
    m_tableName = std::forward<TableNameT>(value);
  }
  template <typename TableNameT = Aws::String>
  GetTableMetadataRequest& WithTableName(TableNameT&& value) {
    SetTableName(std::forward<TableNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the workgroup for which the metadata is being fetched. Required
   * if requesting an IAM Identity Center enabled Glue Data Catalog.</p>
   */
  inline const Aws::String& GetWorkGroup() const { return m_workGroup; }
  inline bool WorkGroupHasBeenSet() const { return m_workGroupHasBeenSet; }
  template <typename WorkGroupT = Aws::String>
  void SetWorkGroup(WorkGroupT&& value) {
    m_workGroupHasBeenSet = true;
    m_workGroup = std::forward<WorkGroupT>(value);
  }
  template <typename WorkGroupT = Aws::String>
  GetTableMetadataRequest& WithWorkGroup(WorkGroupT&& value) {
    SetWorkGroup(std::forward<WorkGroupT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogName;
  bool m_catalogNameHasBeenSet = false;

  Aws::String m_databaseName;
  bool m_databaseNameHasBeenSet = false;

  Aws::String m_tableName;
  bool m_tableNameHasBeenSet = false;

  Aws::String m_workGroup;
  bool m_workGroupHasBeenSet = false;
};

}  // namespace Model
}  // namespace Athena
}  // namespace Aws
