#include "livekit/rtc_error.h"
#include "livekit/helper.h"
#include "livekit/candidate.h"
#include "livekit/media_stream.h"
#include "livekit/rtp_transceiver.h"
#include "livekit/rtp_sender.h"
#include "livekit/rtp_receiver.h"
#include "livekit/data_channel.h"
#include "livekit/jsep.h"
#include "livekit/webrtc.h"
#include "livekit/peer_connection.h"
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <exception>
#include <initializer_list>
#include <iterator>
#include <memory>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 202002L
#include <ranges>
#endif

#ifdef __GNUC__
#pragma GCC diagnostic ignored "-Wmissing-declarations"
#pragma GCC diagnostic ignored "-Wshadow"
#ifdef __clang__
#pragma clang diagnostic ignored "-Wdollar-in-identifier-extension"
#pragma clang diagnostic ignored "-Wreturn-type-c-linkage"
#endif // __clang__
#endif // __GNUC__

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BOX
#define CXXBRIDGE1_RUST_BOX
template <typename T>
class Box final {
public:
  using element_type = T;
  using const_pointer =
      typename std::add_pointer<typename std::add_const<T>::type>::type;
  using pointer = typename std::add_pointer<T>::type;

  Box() = delete;
  Box(Box &&) noexcept;
  ~Box() noexcept;

  explicit Box(const T &);
  explicit Box(T &&);

  Box &operator=(Box &&) & noexcept;

  const T *operator->() const noexcept;
  const T &operator*() const noexcept;
  T *operator->() noexcept;
  T &operator*() noexcept;

  template <typename... Fields>
  static Box in_place(Fields &&...);

  void swap(Box &) noexcept;

  static Box from_raw(T *) noexcept;

  T *into_raw() noexcept;

  /* Deprecated */ using value_type = element_type;

private:
  class uninit;
  class allocation;
  Box(uninit) noexcept;
  void drop() noexcept;

  friend void swap(Box &lhs, Box &rhs) noexcept { lhs.swap(rhs); }

  T *ptr;
};

template <typename T>
class Box<T>::uninit {};

template <typename T>
class Box<T>::allocation {
  static T *alloc() noexcept;
  static void dealloc(T *) noexcept;

public:
  allocation() noexcept : ptr(alloc()) {}
  ~allocation() noexcept {
    if (this->ptr) {
      dealloc(this->ptr);
    }
  }
  T *ptr;
};

template <typename T>
Box<T>::Box(Box &&other) noexcept : ptr(other.ptr) {
  other.ptr = nullptr;
}

template <typename T>
Box<T>::Box(const T &val) {
  allocation alloc;
  ::new (alloc.ptr) T(val);
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::Box(T &&val) {
  allocation alloc;
  ::new (alloc.ptr) T(std::move(val));
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::~Box() noexcept {
  if (this->ptr) {
    this->drop();
  }
}

template <typename T>
Box<T> &Box<T>::operator=(Box &&other) & noexcept {
  if (this->ptr) {
    this->drop();
  }
  this->ptr = other.ptr;
  other.ptr = nullptr;
  return *this;
}

template <typename T>
const T *Box<T>::operator->() const noexcept {
  return this->ptr;
}

template <typename T>
const T &Box<T>::operator*() const noexcept {
  return *this->ptr;
}

template <typename T>
T *Box<T>::operator->() noexcept {
  return this->ptr;
}

template <typename T>
T &Box<T>::operator*() noexcept {
  return *this->ptr;
}

template <typename T>
template <typename... Fields>
Box<T> Box<T>::in_place(Fields &&...fields) {
  allocation alloc;
  auto ptr = alloc.ptr;
  ::new (ptr) T{std::forward<Fields>(fields)...};
  alloc.ptr = nullptr;
  return from_raw(ptr);
}

template <typename T>
void Box<T>::swap(Box &rhs) noexcept {
  using std::swap;
  swap(this->ptr, rhs.ptr);
}

template <typename T>
Box<T> Box<T>::from_raw(T *raw) noexcept {
  Box box = uninit{};
  box.ptr = raw;
  return box;
}

template <typename T>
T *Box<T>::into_raw() noexcept {
  T *raw = this->ptr;
  this->ptr = nullptr;
  return raw;
}

template <typename T>
Box<T>::Box(uninit) noexcept {}
#endif // CXXBRIDGE1_RUST_BOX

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_BITCOPY
#define CXXBRIDGE1_RUST_BITCOPY
constexpr unsafe_bitcopy_t unsafe_bitcopy{};
#endif // CXXBRIDGE1_RUST_BITCOPY

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_FN
#define CXXBRIDGE1_RUST_FN
template <typename Signature>
class Fn;

template <typename Ret, typename... Args>
class Fn<Ret(Args...)> final {
public:
  Ret operator()(Args... args) const noexcept;
  Fn operator*() const noexcept;

private:
  Ret (*trampoline)(Args..., void *fn) noexcept;
  void *fn;
};

template <typename Ret, typename... Args>
Ret Fn<Ret(Args...)>::operator()(Args... args) const noexcept {
  return (*this->trampoline)(std::forward<Args>(args)..., this->fn);
}

template <typename Ret, typename... Args>
Fn<Ret(Args...)> Fn<Ret(Args...)>::operator*() const noexcept {
  return *this;
}
#endif // CXXBRIDGE1_RUST_FN

#ifndef CXXBRIDGE1_RUST_OPAQUE
#define CXXBRIDGE1_RUST_OPAQUE
class Opaque {
public:
  Opaque() = delete;
  Opaque(const Opaque &) = delete;
  ~Opaque() = delete;
};
#endif // CXXBRIDGE1_RUST_OPAQUE

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

#ifndef CXXBRIDGE1_RELOCATABLE
#define CXXBRIDGE1_RELOCATABLE
namespace detail {
template <typename... Ts>
struct make_void {
  using type = void;
};

template <typename... Ts>
using void_t = typename make_void<Ts...>::type;

template <typename Void, template <typename...> class, typename...>
struct detect : std::false_type {};
template <template <typename...> class T, typename... A>
struct detect<void_t<T<A...>>, T, A...> : std::true_type {};

template <template <typename...> class T, typename... A>
using is_detected = detect<void, T, A...>;

template <typename T>
using detect_IsRelocatable = typename T::IsRelocatable;

template <typename T>
struct get_IsRelocatable
    : std::is_same<typename T::IsRelocatable, std::true_type> {};
} // namespace detail

template <typename T>
struct IsRelocatable
    : std::conditional<
          detail::is_detected<detail::detect_IsRelocatable, T>::value,
          detail::get_IsRelocatable<T>,
          std::integral_constant<
              bool, std::is_trivially_move_constructible<T>::value &&
                        std::is_trivially_destructible<T>::value>>::type {};
#endif // CXXBRIDGE1_RELOCATABLE

namespace repr {
struct PtrLen final {
  void *ptr;
  ::std::size_t len;
};
} // namespace repr

namespace detail {
class Fail final {
  ::rust::repr::PtrLen &throw$;
public:
  Fail(::rust::repr::PtrLen &throw$) noexcept : throw$(throw$) {}
  void operator()(char const *) noexcept;
  void operator()(std::string const &) noexcept;
};
} // namespace detail

template <typename T>
union ManuallyDrop {
  T value;
  ManuallyDrop(T &&value) : value(::std::move(value)) {}
  ~ManuallyDrop() {}
};

namespace {
template <typename T, bool = ::rust::detail::is_complete<T>::value>
struct is_destructible : ::std::false_type {};
template <typename T>
struct is_destructible<T, true> : ::std::is_destructible<T> {};
template <typename T>
struct is_destructible<T[], false> : is_destructible<T> {};
template <typename T, bool = ::rust::is_destructible<T>::value>
struct shared_ptr_if_destructible {
  explicit shared_ptr_if_destructible(typename ::std::shared_ptr<T>::element_type *) {}
};
template <typename T>
struct shared_ptr_if_destructible<T, true> : ::std::shared_ptr<T> {
  using ::std::shared_ptr<T>::shared_ptr;
};

template <typename T>
struct IsRelocatableOrArray : IsRelocatable<T> {};
template <typename T, ::std::size_t N>
struct IsRelocatableOrArray<T[N]> : IsRelocatableOrArray<T> {};
} // namespace
} // namespace cxxbridge1

namespace behavior {
class missing {};
missing trycatch(...);

template <typename Try, typename Fail>
static typename ::std::enable_if<::std::is_same<
    decltype(trycatch(::std::declval<Try>(), ::std::declval<Fail>())),
    missing>::value>::type
trycatch(Try &&func, Fail &&fail) noexcept try {
  func();
} catch (::std::exception const &e) {
  fail(e.what());
}
} // namespace behavior
} // namespace rust

#if __cplusplus >= 201402L
#define CXX_DEFAULT_VALUE(value) = value
#else
#define CXX_DEFAULT_VALUE(value)
#endif

namespace livekit {
  enum class PeerConnectionState : ::std::int32_t;
  enum class SignalingState : ::std::int32_t;
  enum class IceConnectionState : ::std::int32_t;
  enum class IceGatheringState : ::std::int32_t;
  enum class ContinualGatheringPolicy : ::std::int32_t;
  enum class IceTransportsType : ::std::int32_t;
  struct RtcOfferAnswerOptions;
  struct IceServer;
  struct RtcConfiguration;
  using PeerConnection = ::livekit::PeerConnection;
  struct PeerContext;
}

namespace livekit {
#ifndef CXXBRIDGE1_ENUM_livekit$PeerConnectionState
#define CXXBRIDGE1_ENUM_livekit$PeerConnectionState
enum class PeerConnectionState : ::std::int32_t {
  New = 0,
  Connecting = 1,
  Connected = 2,
  Disconnected = 3,
  Failed = 4,
  Closed = 5,
};
#endif // CXXBRIDGE1_ENUM_livekit$PeerConnectionState

#ifndef CXXBRIDGE1_ENUM_livekit$SignalingState
#define CXXBRIDGE1_ENUM_livekit$SignalingState
enum class SignalingState : ::std::int32_t {
  Stable = 0,
  HaveLocalOffer = 1,
  HaveLocalPrAnswer = 2,
  HaveRemoteOffer = 3,
  HaveRemotePrAnswer = 4,
  Closed = 5,
};
#endif // CXXBRIDGE1_ENUM_livekit$SignalingState

#ifndef CXXBRIDGE1_ENUM_livekit$IceConnectionState
#define CXXBRIDGE1_ENUM_livekit$IceConnectionState
enum class IceConnectionState : ::std::int32_t {
  IceConnectionNew = 0,
  IceConnectionChecking = 1,
  IceConnectionConnected = 2,
  IceConnectionCompleted = 3,
  IceConnectionFailed = 4,
  IceConnectionDisconnected = 5,
  IceConnectionClosed = 6,
  IceConnectionMax = 7,
};
#endif // CXXBRIDGE1_ENUM_livekit$IceConnectionState

#ifndef CXXBRIDGE1_ENUM_livekit$IceGatheringState
#define CXXBRIDGE1_ENUM_livekit$IceGatheringState
enum class IceGatheringState : ::std::int32_t {
  IceGatheringNew = 0,
  IceGatheringGathering = 1,
  IceGatheringComplete = 2,
};
#endif // CXXBRIDGE1_ENUM_livekit$IceGatheringState

#ifndef CXXBRIDGE1_ENUM_livekit$ContinualGatheringPolicy
#define CXXBRIDGE1_ENUM_livekit$ContinualGatheringPolicy
enum class ContinualGatheringPolicy : ::std::int32_t {
  GatherOnce = 0,
  GatherContinually = 1,
};
#endif // CXXBRIDGE1_ENUM_livekit$ContinualGatheringPolicy

#ifndef CXXBRIDGE1_ENUM_livekit$IceTransportsType
#define CXXBRIDGE1_ENUM_livekit$IceTransportsType
enum class IceTransportsType : ::std::int32_t {
  None = 0,
  Relay = 1,
  NoHost = 2,
  All = 3,
};
#endif // CXXBRIDGE1_ENUM_livekit$IceTransportsType

#ifndef CXXBRIDGE1_STRUCT_livekit$RtcOfferAnswerOptions
#define CXXBRIDGE1_STRUCT_livekit$RtcOfferAnswerOptions
struct RtcOfferAnswerOptions final {
  ::std::int32_t offer_to_receive_video CXX_DEFAULT_VALUE(0);
  ::std::int32_t offer_to_receive_audio CXX_DEFAULT_VALUE(0);
  bool voice_activity_detection CXX_DEFAULT_VALUE(false);
  bool ice_restart CXX_DEFAULT_VALUE(false);
  bool use_rtp_mux CXX_DEFAULT_VALUE(false);
  bool raw_packetization_for_video CXX_DEFAULT_VALUE(false);
  ::std::int32_t num_simulcast_layers CXX_DEFAULT_VALUE(0);
  bool use_obsolete_sctp_sdp CXX_DEFAULT_VALUE(false);

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_livekit$RtcOfferAnswerOptions

#ifndef CXXBRIDGE1_STRUCT_livekit$IceServer
#define CXXBRIDGE1_STRUCT_livekit$IceServer
struct IceServer final {
  ::rust::Vec<::rust::String> urls;
  ::rust::String username;
  ::rust::String password;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_livekit$IceServer

#ifndef CXXBRIDGE1_STRUCT_livekit$RtcConfiguration
#define CXXBRIDGE1_STRUCT_livekit$RtcConfiguration
struct RtcConfiguration final {
  ::rust::Vec<::livekit::IceServer> ice_servers;
  ::livekit::ContinualGatheringPolicy continual_gathering_policy;
  ::livekit::IceTransportsType ice_transport_type;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_livekit$RtcConfiguration

#ifndef CXXBRIDGE1_STRUCT_livekit$PeerContext
#define CXXBRIDGE1_STRUCT_livekit$PeerContext
struct PeerContext final : public ::rust::Opaque {
  ~PeerContext() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_livekit$PeerContext
} // namespace livekit

static_assert(
    ::std::is_base_of<::rust::Opaque, ::livekit::RtpSenderPtr>::value || ::rust::IsRelocatableOrArray<::livekit::RtpSenderPtr>::value,
    "type livekit::RtpSenderPtr should be trivially move constructible and trivially destructible in C++ to be used as a vector element in Vec<RtpSenderPtr> in Rust");
static_assert(
    ::std::is_base_of<::rust::Opaque, ::livekit::RtpReceiverPtr>::value || ::rust::IsRelocatableOrArray<::livekit::RtpReceiverPtr>::value,
    "type livekit::RtpReceiverPtr should be trivially move constructible and trivially destructible in C++ to be used as a vector element in Vec<RtpReceiverPtr> in Rust");
static_assert(
    ::std::is_base_of<::rust::Opaque, ::livekit::RtpTransceiverPtr>::value || ::rust::IsRelocatableOrArray<::livekit::RtpTransceiverPtr>::value,
    "type livekit::RtpTransceiverPtr should be trivially move constructible and trivially destructible in C++ to be used as a vector element in Vec<RtpTransceiverPtr> in Rust");
static_assert(
    ::rust::IsRelocatable<::livekit::DataChannelInit>::value,
    "type livekit::DataChannelInit should be trivially move constructible and trivially destructible in C++ to be used as an argument of `create_data_channel` in Rust");
static_assert(
    ::rust::IsRelocatable<::livekit::RtpTransceiverInit>::value,
    "type livekit::RtpTransceiverInit should be trivially move constructible and trivially destructible in C++ to be used as an argument of `add_transceiver`, `add_transceiver_for_media` in Rust");
static_assert(
    ::rust::IsRelocatable<::livekit::MediaType>::value,
    "type livekit::MediaType should be trivially move constructible and trivially destructible in C++ to be used as an argument of `add_transceiver_for_media` in Rust");

namespace livekit {
extern "C" {
::rust::repr::PtrLen livekit$cxxbridge1$PeerConnection$set_configuration(::livekit::PeerConnection const &self, ::livekit::RtcConfiguration *config) noexcept {
  void (::livekit::PeerConnection::*set_configuration$)(::livekit::RtcConfiguration) const = &::livekit::PeerConnection::set_configuration;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        (self.*set_configuration$)(::std::move(*config));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

void livekit$cxxbridge1$PeerConnection$create_offer(::livekit::PeerConnection const &self, ::livekit::RtcOfferAnswerOptions *options, ::livekit::PeerContext *ctx, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::std::unique_ptr<::livekit::SessionDescription>)> on_success, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)> on_error) noexcept {
  void (::livekit::PeerConnection::*create_offer$)(::livekit::RtcOfferAnswerOptions, ::rust::Box<::livekit::PeerContext>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::std::unique_ptr<::livekit::SessionDescription>)>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)>) const = &::livekit::PeerConnection::create_offer;
  (self.*create_offer$)(::std::move(*options), ::rust::Box<::livekit::PeerContext>::from_raw(ctx), on_success, on_error);
}

void livekit$cxxbridge1$PeerConnection$create_offer$on_success$1(::livekit::PeerContext *ctx, ::livekit::SessionDescription *sdp, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$create_offer$on_success$0(::rust::Box<::livekit::PeerContext> ctx, ::std::unique_ptr<::livekit::SessionDescription> sdp, void *extern$) noexcept {
  livekit$cxxbridge1$PeerConnection$create_offer$on_success$1(ctx.into_raw(), sdp.release(), extern$);
}

void livekit$cxxbridge1$PeerConnection$create_offer$on_error$1(::livekit::PeerContext *ctx, ::livekit::RtcError *error, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$create_offer$on_error$0(::rust::Box<::livekit::PeerContext> ctx, ::livekit::RtcError error, void *extern$) noexcept {
  ::rust::ManuallyDrop<::livekit::RtcError> error$(::std::move(error));
  livekit$cxxbridge1$PeerConnection$create_offer$on_error$1(ctx.into_raw(), &error$.value, extern$);
}

void livekit$cxxbridge1$PeerConnection$create_answer(::livekit::PeerConnection const &self, ::livekit::RtcOfferAnswerOptions *options, ::livekit::PeerContext *ctx, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::std::unique_ptr<::livekit::SessionDescription>)> on_success, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)> on_error) noexcept {
  void (::livekit::PeerConnection::*create_answer$)(::livekit::RtcOfferAnswerOptions, ::rust::Box<::livekit::PeerContext>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::std::unique_ptr<::livekit::SessionDescription>)>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)>) const = &::livekit::PeerConnection::create_answer;
  (self.*create_answer$)(::std::move(*options), ::rust::Box<::livekit::PeerContext>::from_raw(ctx), on_success, on_error);
}

void livekit$cxxbridge1$PeerConnection$create_answer$on_success$1(::livekit::PeerContext *ctx, ::livekit::SessionDescription *sdp, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$create_answer$on_success$0(::rust::Box<::livekit::PeerContext> ctx, ::std::unique_ptr<::livekit::SessionDescription> sdp, void *extern$) noexcept {
  livekit$cxxbridge1$PeerConnection$create_answer$on_success$1(ctx.into_raw(), sdp.release(), extern$);
}

void livekit$cxxbridge1$PeerConnection$create_answer$on_error$1(::livekit::PeerContext *ctx, ::livekit::RtcError *error, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$create_answer$on_error$0(::rust::Box<::livekit::PeerContext> ctx, ::livekit::RtcError error, void *extern$) noexcept {
  ::rust::ManuallyDrop<::livekit::RtcError> error$(::std::move(error));
  livekit$cxxbridge1$PeerConnection$create_answer$on_error$1(ctx.into_raw(), &error$.value, extern$);
}

void livekit$cxxbridge1$PeerConnection$set_local_description(::livekit::PeerConnection const &self, ::livekit::SessionDescription *desc, ::livekit::PeerContext *ctx, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)> on_complete) noexcept {
  void (::livekit::PeerConnection::*set_local_description$)(::std::unique_ptr<::livekit::SessionDescription>, ::rust::Box<::livekit::PeerContext>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)>) const = &::livekit::PeerConnection::set_local_description;
  (self.*set_local_description$)(::std::unique_ptr<::livekit::SessionDescription>(desc), ::rust::Box<::livekit::PeerContext>::from_raw(ctx), on_complete);
}

void livekit$cxxbridge1$PeerConnection$set_local_description$on_complete$1(::livekit::PeerContext *ctx, ::livekit::RtcError *error, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$set_local_description$on_complete$0(::rust::Box<::livekit::PeerContext> ctx, ::livekit::RtcError error, void *extern$) noexcept {
  ::rust::ManuallyDrop<::livekit::RtcError> error$(::std::move(error));
  livekit$cxxbridge1$PeerConnection$set_local_description$on_complete$1(ctx.into_raw(), &error$.value, extern$);
}

void livekit$cxxbridge1$PeerConnection$set_remote_description(::livekit::PeerConnection const &self, ::livekit::SessionDescription *desc, ::livekit::PeerContext *ctx, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)> on_complete) noexcept {
  void (::livekit::PeerConnection::*set_remote_description$)(::std::unique_ptr<::livekit::SessionDescription>, ::rust::Box<::livekit::PeerContext>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)>) const = &::livekit::PeerConnection::set_remote_description;
  (self.*set_remote_description$)(::std::unique_ptr<::livekit::SessionDescription>(desc), ::rust::Box<::livekit::PeerContext>::from_raw(ctx), on_complete);
}

void livekit$cxxbridge1$PeerConnection$set_remote_description$on_complete$1(::livekit::PeerContext *ctx, ::livekit::RtcError *error, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$set_remote_description$on_complete$0(::rust::Box<::livekit::PeerContext> ctx, ::livekit::RtcError error, void *extern$) noexcept {
  ::rust::ManuallyDrop<::livekit::RtcError> error$(::std::move(error));
  livekit$cxxbridge1$PeerConnection$set_remote_description$on_complete$1(ctx.into_raw(), &error$.value, extern$);
}

::rust::repr::PtrLen livekit$cxxbridge1$PeerConnection$add_track(::livekit::PeerConnection const &self, ::std::shared_ptr<::livekit::MediaStreamTrack> *track, ::rust::Vec<::rust::String> const &stream_ids, ::std::shared_ptr<::livekit::RtpSender> *return$) noexcept {
  ::std::shared_ptr<::livekit::RtpSender> (::livekit::PeerConnection::*add_track$)(::std::shared_ptr<::livekit::MediaStreamTrack>, ::rust::Vec<::rust::String> const &) const = &::livekit::PeerConnection::add_track;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::livekit::RtpSender>((self.*add_track$)(::std::move(*track), stream_ids));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen livekit$cxxbridge1$PeerConnection$remove_track(::livekit::PeerConnection const &self, ::std::shared_ptr<::livekit::RtpSender> *sender) noexcept {
  void (::livekit::PeerConnection::*remove_track$)(::std::shared_ptr<::livekit::RtpSender>) const = &::livekit::PeerConnection::remove_track;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        (self.*remove_track$)(::std::move(*sender));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

void livekit$cxxbridge1$PeerConnection$get_stats(::livekit::PeerConnection const &self, ::livekit::PeerContext *ctx, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::rust::String)> on_stats) noexcept {
  void (::livekit::PeerConnection::*get_stats$)(::rust::Box<::livekit::PeerContext>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::rust::String)>) const = &::livekit::PeerConnection::get_stats;
  (self.*get_stats$)(::rust::Box<::livekit::PeerContext>::from_raw(ctx), on_stats);
}

void livekit$cxxbridge1$PeerConnection$get_stats$on_stats$1(::livekit::PeerContext *ctx, ::rust::String *json, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$get_stats$on_stats$0(::rust::Box<::livekit::PeerContext> ctx, ::rust::String json, void *extern$) noexcept {
  livekit$cxxbridge1$PeerConnection$get_stats$on_stats$1(ctx.into_raw(), &json, extern$);
}

::rust::repr::PtrLen livekit$cxxbridge1$PeerConnection$add_transceiver(::livekit::PeerConnection const &self, ::std::shared_ptr<::livekit::MediaStreamTrack> *track, ::livekit::RtpTransceiverInit *init, ::std::shared_ptr<::livekit::RtpTransceiver> *return$) noexcept {
  ::std::shared_ptr<::livekit::RtpTransceiver> (::livekit::PeerConnection::*add_transceiver$)(::std::shared_ptr<::livekit::MediaStreamTrack>, ::livekit::RtpTransceiverInit) const = &::livekit::PeerConnection::add_transceiver;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::livekit::RtpTransceiver>((self.*add_transceiver$)(::std::move(*track), ::std::move(*init)));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

::rust::repr::PtrLen livekit$cxxbridge1$PeerConnection$add_transceiver_for_media(::livekit::PeerConnection const &self, ::livekit::MediaType *media_type, ::livekit::RtpTransceiverInit *init, ::std::shared_ptr<::livekit::RtpTransceiver> *return$) noexcept {
  ::std::shared_ptr<::livekit::RtpTransceiver> (::livekit::PeerConnection::*add_transceiver_for_media$)(::livekit::MediaType, ::livekit::RtpTransceiverInit) const = &::livekit::PeerConnection::add_transceiver_for_media;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::livekit::RtpTransceiver>((self.*add_transceiver_for_media$)(::std::move(*media_type), ::std::move(*init)));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

void livekit$cxxbridge1$PeerConnection$get_senders(::livekit::PeerConnection const &self, ::rust::Vec<::livekit::RtpSenderPtr> *return$) noexcept {
  ::rust::Vec<::livekit::RtpSenderPtr> (::livekit::PeerConnection::*get_senders$)() const = &::livekit::PeerConnection::get_senders;
  new (return$) ::rust::Vec<::livekit::RtpSenderPtr>((self.*get_senders$)());
}

void livekit$cxxbridge1$PeerConnection$get_receivers(::livekit::PeerConnection const &self, ::rust::Vec<::livekit::RtpReceiverPtr> *return$) noexcept {
  ::rust::Vec<::livekit::RtpReceiverPtr> (::livekit::PeerConnection::*get_receivers$)() const = &::livekit::PeerConnection::get_receivers;
  new (return$) ::rust::Vec<::livekit::RtpReceiverPtr>((self.*get_receivers$)());
}

void livekit$cxxbridge1$PeerConnection$get_transceivers(::livekit::PeerConnection const &self, ::rust::Vec<::livekit::RtpTransceiverPtr> *return$) noexcept {
  ::rust::Vec<::livekit::RtpTransceiverPtr> (::livekit::PeerConnection::*get_transceivers$)() const = &::livekit::PeerConnection::get_transceivers;
  new (return$) ::rust::Vec<::livekit::RtpTransceiverPtr>((self.*get_transceivers$)());
}

::rust::repr::PtrLen livekit$cxxbridge1$PeerConnection$create_data_channel(::livekit::PeerConnection const &self, ::rust::String const *label, ::livekit::DataChannelInit *init, ::std::shared_ptr<::livekit::DataChannel> *return$) noexcept {
  ::std::shared_ptr<::livekit::DataChannel> (::livekit::PeerConnection::*create_data_channel$)(::rust::String, ::livekit::DataChannelInit) const = &::livekit::PeerConnection::create_data_channel;
  ::rust::repr::PtrLen throw$;
  ::rust::behavior::trycatch(
      [&] {
        new (return$) ::std::shared_ptr<::livekit::DataChannel>((self.*create_data_channel$)(::rust::String(::rust::unsafe_bitcopy, *label), ::std::move(*init)));
        throw$.ptr = nullptr;
      },
      ::rust::detail::Fail(throw$));
  return throw$;
}

void livekit$cxxbridge1$PeerConnection$add_ice_candidate(::livekit::PeerConnection const &self, ::std::shared_ptr<::livekit::IceCandidate> *candidate, ::livekit::PeerContext *ctx, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)> on_complete) noexcept {
  void (::livekit::PeerConnection::*add_ice_candidate$)(::std::shared_ptr<::livekit::IceCandidate>, ::rust::Box<::livekit::PeerContext>, ::rust::Fn<void(::rust::Box<::livekit::PeerContext>, ::livekit::RtcError)>) const = &::livekit::PeerConnection::add_ice_candidate;
  (self.*add_ice_candidate$)(::std::move(*candidate), ::rust::Box<::livekit::PeerContext>::from_raw(ctx), on_complete);
}

void livekit$cxxbridge1$PeerConnection$add_ice_candidate$on_complete$1(::livekit::PeerContext *ctx, ::livekit::RtcError *error, void *) noexcept;

void livekit$cxxbridge1$PeerConnection$add_ice_candidate$on_complete$0(::rust::Box<::livekit::PeerContext> ctx, ::livekit::RtcError error, void *extern$) noexcept {
  ::rust::ManuallyDrop<::livekit::RtcError> error$(::std::move(error));
  livekit$cxxbridge1$PeerConnection$add_ice_candidate$on_complete$1(ctx.into_raw(), &error$.value, extern$);
}

void livekit$cxxbridge1$PeerConnection$restart_ice(::livekit::PeerConnection const &self) noexcept {
  void (::livekit::PeerConnection::*restart_ice$)() const = &::livekit::PeerConnection::restart_ice;
  (self.*restart_ice$)();
}

::livekit::SessionDescription *livekit$cxxbridge1$PeerConnection$current_local_description(::livekit::PeerConnection const &self) noexcept {
  ::std::unique_ptr<::livekit::SessionDescription> (::livekit::PeerConnection::*current_local_description$)() const = &::livekit::PeerConnection::current_local_description;
  return (self.*current_local_description$)().release();
}

::livekit::SessionDescription *livekit$cxxbridge1$PeerConnection$current_remote_description(::livekit::PeerConnection const &self) noexcept {
  ::std::unique_ptr<::livekit::SessionDescription> (::livekit::PeerConnection::*current_remote_description$)() const = &::livekit::PeerConnection::current_remote_description;
  return (self.*current_remote_description$)().release();
}

::livekit::PeerConnectionState livekit$cxxbridge1$PeerConnection$connection_state(::livekit::PeerConnection const &self) noexcept {
  ::livekit::PeerConnectionState (::livekit::PeerConnection::*connection_state$)() const = &::livekit::PeerConnection::connection_state;
  return (self.*connection_state$)();
}

::livekit::SignalingState livekit$cxxbridge1$PeerConnection$signaling_state(::livekit::PeerConnection const &self) noexcept {
  ::livekit::SignalingState (::livekit::PeerConnection::*signaling_state$)() const = &::livekit::PeerConnection::signaling_state;
  return (self.*signaling_state$)();
}

::livekit::IceGatheringState livekit$cxxbridge1$PeerConnection$ice_gathering_state(::livekit::PeerConnection const &self) noexcept {
  ::livekit::IceGatheringState (::livekit::PeerConnection::*ice_gathering_state$)() const = &::livekit::PeerConnection::ice_gathering_state;
  return (self.*ice_gathering_state$)();
}

::livekit::IceConnectionState livekit$cxxbridge1$PeerConnection$ice_connection_state(::livekit::PeerConnection const &self) noexcept {
  ::livekit::IceConnectionState (::livekit::PeerConnection::*ice_connection_state$)() const = &::livekit::PeerConnection::ice_connection_state;
  return (self.*ice_connection_state$)();
}

void livekit$cxxbridge1$PeerConnection$close(::livekit::PeerConnection const &self) noexcept {
  void (::livekit::PeerConnection::*close$)() const = &::livekit::PeerConnection::close;
  (self.*close$)();
}

void livekit$cxxbridge1$_shared_peer_connection(::std::shared_ptr<::livekit::PeerConnection> *return$) noexcept {
  ::std::shared_ptr<::livekit::PeerConnection> (*_shared_peer_connection$)() = ::livekit::_shared_peer_connection;
  new (return$) ::std::shared_ptr<::livekit::PeerConnection>(_shared_peer_connection$());
}
::std::size_t livekit$cxxbridge1$PeerContext$operator$sizeof() noexcept;
::std::size_t livekit$cxxbridge1$PeerContext$operator$alignof() noexcept;
} // extern "C"

::std::size_t PeerContext::layout::size() noexcept {
  return livekit$cxxbridge1$PeerContext$operator$sizeof();
}

::std::size_t PeerContext::layout::align() noexcept {
  return livekit$cxxbridge1$PeerContext$operator$alignof();
}
} // namespace livekit

extern "C" {
void cxxbridge1$rust_vec$livekit$IceServer$new(::rust::Vec<::livekit::IceServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$livekit$IceServer$drop(::rust::Vec<::livekit::IceServer> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$livekit$IceServer$len(::rust::Vec<::livekit::IceServer> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$livekit$IceServer$capacity(::rust::Vec<::livekit::IceServer> const *ptr) noexcept;
::livekit::IceServer const *cxxbridge1$rust_vec$livekit$IceServer$data(::rust::Vec<::livekit::IceServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$livekit$IceServer$reserve_total(::rust::Vec<::livekit::IceServer> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$livekit$IceServer$set_len(::rust::Vec<::livekit::IceServer> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$livekit$IceServer$truncate(::rust::Vec<::livekit::IceServer> *ptr, ::std::size_t len) noexcept;

::livekit::PeerContext *cxxbridge1$box$livekit$PeerContext$alloc() noexcept;
void cxxbridge1$box$livekit$PeerContext$dealloc(::livekit::PeerContext *) noexcept;
void cxxbridge1$box$livekit$PeerContext$drop(::rust::Box<::livekit::PeerContext> *ptr) noexcept;

static_assert(sizeof(::std::shared_ptr<::livekit::PeerConnection>) == 2 * sizeof(void *), "");
static_assert(alignof(::std::shared_ptr<::livekit::PeerConnection>) == alignof(void *), "");
void cxxbridge1$shared_ptr$livekit$PeerConnection$null(::std::shared_ptr<::livekit::PeerConnection> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::livekit::PeerConnection>();
}
bool cxxbridge1$shared_ptr$livekit$PeerConnection$raw(::std::shared_ptr<::livekit::PeerConnection> *ptr, ::std::shared_ptr<::livekit::PeerConnection>::element_type *raw) noexcept {
  ::new (ptr) ::rust::shared_ptr_if_destructible<::livekit::PeerConnection>(raw);
  return ::rust::is_destructible<::livekit::PeerConnection>::value;
}
void cxxbridge1$shared_ptr$livekit$PeerConnection$clone(::std::shared_ptr<::livekit::PeerConnection> const &self, ::std::shared_ptr<::livekit::PeerConnection> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::livekit::PeerConnection>(self);
}
::std::shared_ptr<::livekit::PeerConnection>::element_type const *cxxbridge1$shared_ptr$livekit$PeerConnection$get(::std::shared_ptr<::livekit::PeerConnection> const &self) noexcept {
  return self.get();
}
void cxxbridge1$shared_ptr$livekit$PeerConnection$drop(::std::shared_ptr<::livekit::PeerConnection> *self) noexcept {
  self->~shared_ptr();
}
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
Vec<::livekit::IceServer>::Vec() noexcept {
  cxxbridge1$rust_vec$livekit$IceServer$new(this);
}
template <>
void Vec<::livekit::IceServer>::drop() noexcept {
  return cxxbridge1$rust_vec$livekit$IceServer$drop(this);
}
template <>
::std::size_t Vec<::livekit::IceServer>::size() const noexcept {
  return cxxbridge1$rust_vec$livekit$IceServer$len(this);
}
template <>
::std::size_t Vec<::livekit::IceServer>::capacity() const noexcept {
  return cxxbridge1$rust_vec$livekit$IceServer$capacity(this);
}
template <>
::livekit::IceServer const *Vec<::livekit::IceServer>::data() const noexcept {
  return cxxbridge1$rust_vec$livekit$IceServer$data(this);
}
template <>
void Vec<::livekit::IceServer>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$livekit$IceServer$reserve_total(this, new_cap);
}
template <>
void Vec<::livekit::IceServer>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$livekit$IceServer$set_len(this, len);
}
template <>
void Vec<::livekit::IceServer>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$livekit$IceServer$truncate(this, len);
}
template <>
::livekit::PeerContext *Box<::livekit::PeerContext>::allocation::alloc() noexcept {
  return cxxbridge1$box$livekit$PeerContext$alloc();
}
template <>
void Box<::livekit::PeerContext>::allocation::dealloc(::livekit::PeerContext *ptr) noexcept {
  cxxbridge1$box$livekit$PeerContext$dealloc(ptr);
}
template <>
void Box<::livekit::PeerContext>::drop() noexcept {
  cxxbridge1$box$livekit$PeerContext$drop(this);
}
} // namespace cxxbridge1
} // namespace rust
