#ifndef GEN_UI_GFX_X_SHAPE_H_
#define GEN_UI_GFX_X_SHAPE_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"
#include "xproto.h"

namespace x11 {

class Connection;

class COMPONENT_EXPORT(X11) Shape {
  public:
  static constexpr unsigned major_version = 1;
  static constexpr unsigned minor_version = 1;

  Shape(Connection* connection,
      const x11::QueryExtensionReply& info);

  uint8_t present() const {
    return info_.present;
  }
  uint8_t major_opcode() const {
    return info_.major_opcode;
  }
  uint8_t first_event() const {
    return info_.first_event;
  }
  uint8_t first_error() const {
    return info_.first_error;
  }

  Connection* connection() const { return connection_; }

  enum class Operation : uint8_t {};

  enum class Kind : uint8_t {};

  enum class So : int {
    Set = 0,
    Union = 1,
    Intersect = 2,
    Subtract = 3,
    Invert = 4,
  };

  enum class Sk : int {
    Bounding = 0,
    Clip = 1,
    Input = 2,
  };

  struct NotifyEvent {
    static constexpr int type_id = 14;
    static constexpr uint8_t opcode = 0;
    bool send_event{};
    Sk shape_kind{};
    uint16_t sequence{};
    Window affected_window{};
    int16_t extents_x{};
    int16_t extents_y{};
    uint16_t extents_width{};
    uint16_t extents_height{};
    Time server_time{};
    uint8_t shaped{};

    x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&affected_window); }
  };

  struct QueryVersionRequest {
  };

  struct QueryVersionReply {
    uint16_t sequence{};
    uint16_t major_version{};
    uint16_t minor_version{};
  };

  using QueryVersionResponse = Response<QueryVersionReply>;

  Future<QueryVersionReply> QueryVersion(
      const QueryVersionRequest& request);

  struct RectanglesRequest {
    So operation{};
    Sk destination_kind{};
    ClipOrdering ordering{};
    Window destination_window{};
    int16_t x_offset{};
    int16_t y_offset{};
    std::vector<Rectangle> rectangles{};
  };

  using RectanglesResponse = Response<void>;

  Future<void> Rectangles(
      const RectanglesRequest& request);

  struct MaskRequest {
    So operation{};
    Sk destination_kind{};
    Window destination_window{};
    int16_t x_offset{};
    int16_t y_offset{};
    Pixmap source_bitmap{};
  };

  using MaskResponse = Response<void>;

  Future<void> Mask(
      const MaskRequest& request);

  struct CombineRequest {
    So operation{};
    Sk destination_kind{};
    Sk source_kind{};
    Window destination_window{};
    int16_t x_offset{};
    int16_t y_offset{};
    Window source_window{};
  };

  using CombineResponse = Response<void>;

  Future<void> Combine(
      const CombineRequest& request);

  struct OffsetRequest {
    Sk destination_kind{};
    Window destination_window{};
    int16_t x_offset{};
    int16_t y_offset{};
  };

  using OffsetResponse = Response<void>;

  Future<void> Offset(
      const OffsetRequest& request);

  struct QueryExtentsRequest {
    Window destination_window{};
  };

  struct QueryExtentsReply {
    uint16_t sequence{};
    uint8_t bounding_shaped{};
    uint8_t clip_shaped{};
    int16_t bounding_shape_extents_x{};
    int16_t bounding_shape_extents_y{};
    uint16_t bounding_shape_extents_width{};
    uint16_t bounding_shape_extents_height{};
    int16_t clip_shape_extents_x{};
    int16_t clip_shape_extents_y{};
    uint16_t clip_shape_extents_width{};
    uint16_t clip_shape_extents_height{};
  };

  using QueryExtentsResponse = Response<QueryExtentsReply>;

  Future<QueryExtentsReply> QueryExtents(
      const QueryExtentsRequest& request);

  struct SelectInputRequest {
    Window destination_window{};
    uint8_t enable{};
  };

  using SelectInputResponse = Response<void>;

  Future<void> SelectInput(
      const SelectInputRequest& request);

  struct InputSelectedRequest {
    Window destination_window{};
  };

  struct InputSelectedReply {
    uint8_t enabled{};
    uint16_t sequence{};
  };

  using InputSelectedResponse = Response<InputSelectedReply>;

  Future<InputSelectedReply> InputSelected(
      const InputSelectedRequest& request);

  struct GetRectanglesRequest {
    Window window{};
    Sk source_kind{};
  };

  struct GetRectanglesReply {
    ClipOrdering ordering{};
    uint16_t sequence{};
    std::vector<Rectangle> rectangles{};
  };

  using GetRectanglesResponse = Response<GetRectanglesReply>;

  Future<GetRectanglesReply> GetRectangles(
      const GetRectanglesRequest& request);

  private:
  x11::Connection* const connection_;
  x11::QueryExtensionReply info_{};
};

}  // namespace x11

inline constexpr x11::Shape::So operator|(
    x11::Shape::So l, x11::Shape::So r) {
  using T = std::underlying_type_t<x11::Shape::So>;
  return static_cast<x11::Shape::So>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Shape::So operator&(
    x11::Shape::So l, x11::Shape::So r) {
  using T = std::underlying_type_t<x11::Shape::So>;
  return static_cast<x11::Shape::So>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Shape::Sk operator|(
    x11::Shape::Sk l, x11::Shape::Sk r) {
  using T = std::underlying_type_t<x11::Shape::Sk>;
  return static_cast<x11::Shape::Sk>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Shape::Sk operator&(
    x11::Shape::Sk l, x11::Shape::Sk r) {
  using T = std::underlying_type_t<x11::Shape::Sk>;
  return static_cast<x11::Shape::Sk>(
      static_cast<T>(l) & static_cast<T>(r));
}


#endif  // GEN_UI_GFX_X_SHAPE_H_
