// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F340

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_256_0_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x2b06876cu, 0xba6b15afu, 0, 155 }, // 2b06876cba6b15af = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xfbb53856u, 0xe03852b6u, 0, 6 }, // fbb53856e03852b6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x4badb33cu, 0x3467fb4du, 0, 69 }, // 4badb33c3467fb4d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x1ae11feau, 0xbf841d31u, 23, 155 }, // 1ae11feabf841d31 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x42fba6a2u, 0xd3f52be3u, 23, 138 }, // 42fba6a2d3f52be3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x2f445b87u, 0x445f826au, 23, 69 }, // 2f445b87445f826a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x7b5876f9u, 0xa62579feu, 63, 155 }, // 7b5876f9a62579fe = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x364ef992u, 0xc660cb5fu, 63, 138 }, // 364ef992c660cb5f = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_32__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{4,2,0,1,5,1,2},
 {7,2,1,1,5,2,1},
 {5,1,1,1,1,1,3},
 {5,0,1,1,1,1,1},
 {5,1,1,1,1,1,1},
 {5,1,1,1,1,1,1},
 {5,1,1,1,1,1,6}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F340 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 256
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// vim: set fileencoding=utf-8

