// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.
#pragma once

#include "core/providers/rocm/shared_inc/fpgeneric.h"
#include "core/providers/cpu/tensor/utils.h"
#include "contrib_ops/cpu/transformers/generation_shared.h"
#include "core/providers/rocm/math/softmax.h"

#ifdef DEBUG_GENERATION
#include <iostream>
#endif

using onnxruntime::rocm::dispatch_blockwise_softmax_forward;
using onnxruntime::rocm::ToHipType;

namespace onnxruntime {
namespace contrib {
namespace SamplingCudaHelper {

template <typename T>
Status Sample(AllocatorPtr& allocator,
              Stream* stream,
              gsl::span<T>& next_token_scores,
              transformers::ISamplingState<T>* sampling_state,
              transformers::IGreedySearchState<T>* greedy_state,
              const transformers::IGenerationParameters* parameters,
              int step,
              const IConsoleDumper* dumper) {
  ORT_UNUSED_PARAMETER(dumper);
  typedef typename ToHipType<T>::MappedType HipT;

  auto hip_stream = static_cast<hipStream_t>(stream->GetHandle());

  gsl::span<int>& d_index_in = sampling_state->d_index_in;
  gsl::span<int>& d_offset = sampling_state->d_offset;

  BufferUniquePtr& storage_buffer = sampling_state->storage_buffer;
  size_t& temp_storage_bytes = sampling_state->temp_storage_bytes;

  bool is_descending = parameters->custom_sampling;
  if (step == 1) {
    rocm::GetTempStorageSize<HipT>(reinterpret_cast<HipT*>(next_token_scores.data()),
                                    d_index_in.data(),
                                    d_offset.data(),
                                    parameters->batch_size * parameters->vocab_size,
                                    parameters->batch_size,
                                    hip_stream,
                                    is_descending,
                                    temp_storage_bytes);

    rocm::LaunchSetupParamsKernel(d_index_in.data(),
                                  d_offset.data(),
                                  parameters->batch_size,
                                  parameters->vocab_size,
                                  hip_stream);

#ifdef DEBUG_GENERATION
    dumper->Print("d_offset_buffer", d_offset.data(), parameters->batch_size + 1, 1);
#endif

    void* temp_storage = allocator->Alloc(sampling_state->temp_storage_bytes);
    BufferUniquePtr temp_storage_buffer(temp_storage, BufferDeleter(allocator));
    storage_buffer = std::move(temp_storage_buffer);
  }

  gsl::span<T>& d_sorted_score = sampling_state->d_sorted_score;
  gsl::span<int>& d_index_out = sampling_state->d_index_out;

#ifdef DEBUG_GENERATION
  dumper->Print("temp_storage_bytes", sampling_state->temp_storage_bytes, true);
#endif

  rocm::LaunchSortPairs<HipT>(storage_buffer.get(),
                               temp_storage_bytes,
                               reinterpret_cast<HipT*>(next_token_scores.data()),
                               reinterpret_cast<HipT*>(d_sorted_score.data()),
                               d_index_in.data(),
                               d_index_out.data(),
                               parameters->batch_size * parameters->vocab_size,
                               parameters->batch_size,
                               d_offset.data(),
                               hip_stream,
                               is_descending);

#ifdef DEBUG_GENERATION
  dumper->Print("d_sorted_score_buffer",
                reinterpret_cast<T*>(d_sorted_score.data()),
                parameters->batch_size,
                parameters->vocab_size);
  dumper->Print("d_index_buffer_in", d_index_in.data(), parameters->batch_size, parameters->vocab_size);
  dumper->Print("d_index_buffer_out", d_index_out.data(), parameters->batch_size, parameters->vocab_size);
#endif

  gsl::span<float>& d_sorted_softmaxed_score = sampling_state->d_sorted_softmaxed_score;
  ORT_RETURN_IF_ERROR((dispatch_blockwise_softmax_forward<HipT, float, float, false>(stream,
                                                                                      d_sorted_softmaxed_score.data(),
                                                                                      reinterpret_cast<HipT*>(d_sorted_score.data()),
                                                                                      parameters->vocab_size,
                                                                                      parameters->vocab_size,
                                                                                      parameters->vocab_size,
                                                                                      parameters->batch_size)));

#ifdef DEBUG_GENERATION
  dumper->Print("d_sorted_softmaxed_score_buffer",
                d_sorted_softmaxed_score.data(),
                parameters->batch_size,
                parameters->vocab_size);
#endif

  rocm::LaunchFilterLogitsKernel<HipT>(d_sorted_softmaxed_score.data(),
                                        d_index_out.data(),
                                        reinterpret_cast<HipT*>(next_token_scores.data()),
                                        parameters->top_p,
                                        parameters->filter_value,
                                        parameters->min_tokens_to_keep,
                                        parameters->batch_size,
                                        parameters->vocab_size,
                                        hip_stream,
                                        is_descending);

#ifdef DEBUG_GENERATION
  dumper->Print("next_token_scores after filtering logits",
                reinterpret_cast<T*>(next_token_scores.data()),
                parameters->batch_size,
                parameters->vocab_size);
#endif

  gsl::span<float>& d_softmaxed_score = sampling_state->d_softmaxed_score;
  ORT_RETURN_IF_ERROR((dispatch_blockwise_softmax_forward<HipT, float, float, false>(stream,
                                                                                      d_softmaxed_score.data(),
                                                                                      reinterpret_cast<HipT*>(next_token_scores.data()),
                                                                                      parameters->vocab_size,
                                                                                      parameters->vocab_size,
                                                                                      parameters->vocab_size,
                                                                                      parameters->batch_size)));

#ifdef DEBUG_GENERATION
  dumper->Print("d_softmaxed_score_buffer",
                d_softmaxed_score.data(),
                parameters->batch_size,
                parameters->vocab_size);
#endif

  // Multinomial sampling
  gsl::span<float>& d_sampled = sampling_state->d_sampled;
  gsl::span<float>& h_sampled_all = sampling_state->h_sampled_all;
  size_t sample_offset = (static_cast<size_t>(step) - 1) * static_cast<size_t>(parameters->batch_size);
  HIP_RETURN_IF_ERROR(hipMemcpyAsync(d_sampled.data(),
                                       h_sampled_all.data() + sample_offset,
                                       sizeof(float) * parameters->batch_size,
                                       hipMemcpyHostToDevice,
                                       hip_stream));

#ifdef DEBUG_GENERATION
  dumper->Print("d_sampled", d_sampled.data(), parameters->batch_size, 1);
#endif

  gsl::span<int32_t>& d_indices = sampling_state->d_indices;
  gsl::span<int>& presence_mask = sampling_state->d_presence_mask;
  rocm::TorchMultinomialKernelLauncher(d_softmaxed_score.data(),
                                       d_sampled.data(),
                                       d_indices.data(),
                                       parameters->batch_size,
                                       parameters->vocab_size,
                                       presence_mask.data(),
                                       hip_stream);

#ifdef DEBUG_GENERATION
  dumper->Print("d_indices", d_indices.data(), parameters->batch_size, 1);
#endif

  HIP_RETURN_IF_ERROR(hipMemcpyAsync(greedy_state->next_tokens.data(),
                                       sampling_state->d_indices.data(),
                                       greedy_state->next_tokens.size_bytes(),
                                       hipMemcpyDeviceToHost,
                                       hip_stream));

  HIP_RETURN_IF_ERROR(hipMemcpyAsync(sampling_state->h_softmaxed_score.data(),
                                       sampling_state->d_softmaxed_score.data(),
                                       sampling_state->h_softmaxed_score.size_bytes(),
                                       hipMemcpyDeviceToHost,
                                       hip_stream));

  HIP_RETURN_IF_ERROR(hipStreamSynchronize(hip_stream));

  return Status::OK();
}

}  // namespace SamplingCudaHelper
}  // namespace contrib
}  // namespace onnxruntime
