/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef dom_ipc_MemMapSnapshot_h
#define dom_ipc_MemMapSnapshot_h

#include "ErrorList.h"
#include "mozilla/Attributes.h"
#include "mozilla/RangedPtr.h"
#include "mozilla/Result.h"
#include "mozilla/ipc/SharedMemoryMapping.h"

namespace mozilla::ipc {

/**
 * A helper class for creating a read-only snapshot of memory-mapped data.
 *
 * The Init() method initializes a read-write memory mapped region of the given
 * size, which can be initialized with arbitrary data. The Finalize() method
 * remaps that region as read-only (and backs it with a read-only file
 * descriptor), and returns a handle to it.
 *
 * The file descriptor for the resulting AutoMemMap can be shared among
 * processes, to safely access a shared, read-only copy of the data snapshot.
 */
class MOZ_RAII MemMapSnapshot {
 public:
  Result<Ok, nsresult> Init(size_t aSize);
  Result<ReadOnlySharedMemoryHandle, nsresult> Finalize();

  template <typename T>
  RangedPtr<T> Get() {
    MOZ_ASSERT(mMem);
    auto span = mMem.DataAsSpan<T>();
    return {span.data(), span.size()};
  }

 private:
  FreezableSharedMemoryMapping mMem;
};

}  // namespace mozilla::ipc

#endif  // dom_ipc_MemMapSnapshot_h
