package nl

import (
	"fmt"
	"unsafe"
)

const (
	SizeofBridgeVlanInfo = 0x04
	SizeofTunnelMsg      = 0x08
)

/* Bridge Flags */
const (
	BRIDGE_FLAGS_MASTER = iota + 1 /* Bridge command to/from master */
	BRIDGE_FLAGS_SELF              /* Bridge command to/from lowerdev */
)

/* Bridge management nested attributes
 * [IFLA_AF_SPEC] = {
 *     [IFLA_BRIDGE_FLAGS]
 *     [IFLA_BRIDGE_MODE]
 *     [IFLA_BRIDGE_VLAN_INFO]
 * }
 */
const (
	IFLA_BRIDGE_FLAGS = iota
	IFLA_BRIDGE_MODE
	IFLA_BRIDGE_VLAN_INFO
	IFLA_BRIDGE_VLAN_TUNNEL_INFO
)

const (
	IFLA_BRIDGE_VLAN_TUNNEL_UNSPEC = iota
	IFLA_BRIDGE_VLAN_TUNNEL_ID
	IFLA_BRIDGE_VLAN_TUNNEL_VID
	IFLA_BRIDGE_VLAN_TUNNEL_FLAGS
)

const (
	BRIDGE_VLAN_INFO_MASTER = 1 << iota
	BRIDGE_VLAN_INFO_PVID
	BRIDGE_VLAN_INFO_UNTAGGED
	BRIDGE_VLAN_INFO_RANGE_BEGIN
	BRIDGE_VLAN_INFO_RANGE_END
)

// struct bridge_vlan_info {
//   __u16 flags;
//   __u16 vid;
// };

type TunnelInfo struct {
	TunId uint32
	Vid   uint16
}

type BridgeVlanInfo struct {
	Flags uint16
	Vid   uint16
}

func (b *BridgeVlanInfo) Serialize() []byte {
	return (*(*[SizeofBridgeVlanInfo]byte)(unsafe.Pointer(b)))[:]
}

func DeserializeBridgeVlanInfo(b []byte) *BridgeVlanInfo {
	return (*BridgeVlanInfo)(unsafe.Pointer(&b[0:SizeofBridgeVlanInfo][0]))
}

func (b *BridgeVlanInfo) PortVID() bool {
	return b.Flags&BRIDGE_VLAN_INFO_PVID > 0
}

func (b *BridgeVlanInfo) EngressUntag() bool {
	return b.Flags&BRIDGE_VLAN_INFO_UNTAGGED > 0
}

func (b *BridgeVlanInfo) String() string {
	return fmt.Sprintf("%+v", *b)
}

/* New extended info filters for IFLA_EXT_MASK */
const (
	RTEXT_FILTER_VF = 1 << iota
	RTEXT_FILTER_BRVLAN
	RTEXT_FILTER_BRVLAN_COMPRESSED
)

/* VXLAN VNI filter attributes */
const (
	VXLAN_VNIFILTER_UNSPEC = iota
	VXLAN_VNIFILTER_ENTRY
)

const (
	VXLAN_VNIFILTER_ENTRY_UNSPEC = iota
	VXLAN_VNIFILTER_ENTRY_START
	VXLAN_VNIFILTER_ENTRY_END
)

// struct tunnel_msg {
//   __u8 family;
//   __u8 flags;
//   __u16 reserved2;
//   __u32 ifindex;
// };
type TunnelMsg struct {
	Family   uint8
	Flags    uint8
	Reserved uint16
	Ifindex  uint32
}

func NewTunnelMsg(family uint8, ifindex int) *TunnelMsg {
	return &TunnelMsg{
		Family:  family,
		Ifindex: uint32(ifindex),
	}
}

func (msg *TunnelMsg) Len() int {
	return SizeofTunnelMsg
}

func (msg *TunnelMsg) Serialize() []byte {
	return (*(*[SizeofTunnelMsg]byte)(unsafe.Pointer(msg)))[:]
}

func DeserializeTunnelMsg(b []byte) *TunnelMsg {
	return (*TunnelMsg)(unsafe.Pointer(&b[0:SizeofTunnelMsg][0]))
}

type BridgeVniInfo struct {
	Vni    uint32
	VniEnd uint32
}

func (b *BridgeVniInfo) String() string {
	return fmt.Sprintf("%+v", *b)
}

func DeserializeBridgeVniInfo(b []byte) (*BridgeVniInfo, error) {
	vni := &BridgeVniInfo{}
	attrs, err := ParseRouteAttr(b)
	if err != nil {
		return nil, err
	}
	for _, attr := range attrs {
		switch attr.Attr.Type {
		case VXLAN_VNIFILTER_ENTRY_START:
			vni.Vni = NativeEndian().Uint32(attr.Value)
		case VXLAN_VNIFILTER_ENTRY_END:
			vni.VniEnd = NativeEndian().Uint32(attr.Value)
		}
	}
	return vni, nil
}
