use crate::settings::GFontStyle;
use glib::{Object, Properties, prelude::*, subclass};
use gtk4::{
    Accessible, Box, Buildable, CompositeTemplate, ConstraintTarget, IconSize, Image, Stack, Widget,
    subclass::prelude::*,
};
use libadwaita::{Bin, ViewStack, ViewSwitcher, prelude::*, subclass::prelude::*};
use std::cell::Cell;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::FontStyle)]
    #[template(file = "data/resources/ui_templates/article_view/font_style.blp")]
    pub struct FontStyle {
        #[template_child]
        pub switcher: TemplateChild<ViewSwitcher>,
        #[template_child]
        pub stack: TemplateChild<ViewStack>,

        #[property(get, set, name = "font-style", builder(GFontStyle::SansSerif))]
        pub font_style: Cell<GFontStyle>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for FontStyle {
        const NAME: &'static str = "FontStyle";
        type Type = super::FontStyle;
        type ParentType = Bin;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for FontStyle {
        fn constructed(&self) {
            let mut child = self.switcher.first_child();

            while let Some(widget) = child {
                Self::increase_icon_size(&widget);
                child = widget.next_sibling();
            }

            self.obj()
                .bind_property("font-style", &*self.stack, "visible_child_name")
                .transform_to(|_binding, font_style: GFontStyle| {
                    Some(match font_style {
                        GFontStyle::SansSerif => "sans",
                        GFontStyle::Serif => "serif",
                    })
                })
                .transform_from(|_binding, font_style: String| {
                    Some(match font_style.as_str() {
                        "serif" => GFontStyle::Serif,
                        _ => GFontStyle::SansSerif,
                    })
                })
                .sync_create()
                .bidirectional()
                .build();
        }
    }

    impl WidgetImpl for FontStyle {}

    impl BinImpl for FontStyle {}

    impl FontStyle {
        fn increase_icon_size(widget: &Widget) {
            if let Some(stack) = widget.first_child().and_downcast_ref::<Stack>()
                && let Some(vertical_box) = stack
                    .first_child()
                    .and_then(|child| child.next_sibling())
                    .and_downcast_ref::<Box>()
                && let Some(image) = vertical_box
                    .first_child()
                    .and_then(|indicator_bin| indicator_bin.first_child())
                    .and_then(|gizmo| gizmo.next_sibling())
                    .and_then(|gizmo| gizmo.next_sibling())
                    .and_downcast_ref::<Image>()
            {
                image.set_icon_size(IconSize::Large);
            }
        }
    }
}

glib::wrapper! {
    pub struct FontStyle(ObjectSubclass<imp::FontStyle>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for FontStyle {
    fn default() -> Self {
        Object::new()
    }
}
