﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/qconnect/QConnect_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace QConnect {
namespace Model {

/**
 * <p>The configuration information of the grouping of Amazon Q in Connect
 * users.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/qconnect-2020-10-19/GroupingConfiguration">AWS
 * API Reference</a></p>
 */
class GroupingConfiguration {
 public:
  AWS_QCONNECT_API GroupingConfiguration() = default;
  AWS_QCONNECT_API GroupingConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_QCONNECT_API GroupingConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_QCONNECT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The criteria used for grouping Amazon Q in Connect users.</p> <p>The
   * following is the list of supported criteria values.</p> <ul> <li> <p>
   * <code>RoutingProfileArn</code>: Grouping the users by their <a
   * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_RoutingProfile.html">Amazon
   * Connect routing profile ARN</a>. User should have <a
   * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchRoutingProfiles.html">SearchRoutingProfile</a>
   * and <a
   * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeRoutingProfile.html">DescribeRoutingProfile</a>
   * permissions when setting criteria to this value.</p> </li> </ul>
   */
  inline const Aws::String& GetCriteria() const { return m_criteria; }
  inline bool CriteriaHasBeenSet() const { return m_criteriaHasBeenSet; }
  template <typename CriteriaT = Aws::String>
  void SetCriteria(CriteriaT&& value) {
    m_criteriaHasBeenSet = true;
    m_criteria = std::forward<CriteriaT>(value);
  }
  template <typename CriteriaT = Aws::String>
  GroupingConfiguration& WithCriteria(CriteriaT&& value) {
    SetCriteria(std::forward<CriteriaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of values that define different groups of Amazon Q in Connect
   * users.</p> <ul> <li> <p>When setting <code>criteria</code> to
   * <code>RoutingProfileArn</code>, you need to provide a list of ARNs of <a
   * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_RoutingProfile.html">Amazon
   * Connect routing profiles</a> as values of this parameter.</p> </li> </ul>
   */
  inline const Aws::Vector<Aws::String>& GetValues() const { return m_values; }
  inline bool ValuesHasBeenSet() const { return m_valuesHasBeenSet; }
  template <typename ValuesT = Aws::Vector<Aws::String>>
  void SetValues(ValuesT&& value) {
    m_valuesHasBeenSet = true;
    m_values = std::forward<ValuesT>(value);
  }
  template <typename ValuesT = Aws::Vector<Aws::String>>
  GroupingConfiguration& WithValues(ValuesT&& value) {
    SetValues(std::forward<ValuesT>(value));
    return *this;
  }
  template <typename ValuesT = Aws::String>
  GroupingConfiguration& AddValues(ValuesT&& value) {
    m_valuesHasBeenSet = true;
    m_values.emplace_back(std::forward<ValuesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_criteria;
  bool m_criteriaHasBeenSet = false;

  Aws::Vector<Aws::String> m_values;
  bool m_valuesHasBeenSet = false;
};

}  // namespace Model
}  // namespace QConnect
}  // namespace Aws
