/*
 * Copyright (C) 2018-2025 Intel Corporation
 *
 * SPDX-License-Identifier: MIT
 *
 */

#pragma once
#include "shared/source/gmm_helper/gmm_resource_usage_type.h"

#include <cstdint>
#include <memory>
#include <string>
#include <vector>

namespace NEO {
enum class ImagePlane;
struct HardwareInfo;
struct ImageInfo;
struct StorageInfo;
class GmmResourceInfo;
class GmmHelper;

template <typename OverriddenType>
struct Overrider {
    bool enableOverride{false};
    OverriddenType value;

    void doOverride(OverriddenType &variable) const {
        if (enableOverride) {
            variable = value;
        }
    }
};

struct GmmRequirements {
    bool preferCompressed;
    bool allowLargePages;
    Overrider<bool> overriderCacheable;
    Overrider<bool> overriderPreferNoCpuAccess;
};

class Gmm {
  public:
    virtual ~Gmm();
    Gmm() = delete;
    Gmm(GmmHelper *gmmHelper, ImageInfo &inputOutputImgInfo, const StorageInfo &storageInfo, bool preferCompressed);
    Gmm(GmmHelper *gmmHelper, const void *alignedPtr, size_t alignedSize, size_t alignment,
        GmmResourceUsageType gmmResourceUsage, const StorageInfo &storageInfo, const GmmRequirements &gmmRequirements);
    Gmm(GmmHelper *gmmHelper, GmmResourceInfo *inputGmm);
    Gmm(GmmHelper *gmmHelper, GmmResourceInfo *inputGmm, bool openingHandle);

    void queryImageParams(ImageInfo &inputOutputImgInfo);

    void applyAuxFlagsForBuffer(bool preferCompression);
    void applyExtraAuxInitFlag();
    void applyMemoryFlags(const StorageInfo &storageInfo);
    void applyAppResource(const StorageInfo &storageInfo);

    bool unifiedAuxTranslationCapable() const;
    bool hasMultisampleControlSurface() const;

    GmmHelper *getGmmHelper() const;

    uint32_t queryQPitch();
    void updateImgInfoAndDesc(ImageInfo &imgInfo, uint32_t arrayIndex, ImagePlane yuvPlaneType);
    void updateOffsetsInImgInfo(ImageInfo &imgInfo, uint32_t arrayIndex);
    uint8_t resourceCopyBlt(void *sys, void *gpu, uint32_t pitch, uint32_t height, unsigned char upload, ImagePlane plane);

    uint32_t getUnifiedAuxPitchTiles();
    uint32_t getAuxQPitch();
    bool getPreferNoCpuAccess() const { return preferNoCpuAccess; }

    std::vector<uint8_t> resourceParamsData;
    std::unique_ptr<GmmResourceInfo> gmmResourceInfo;

    std::string getUsageTypeString();
    void setCompressionEnabled(bool compresionEnabled) { this->compressionEnabled = compresionEnabled; }
    bool isCompressionEnabled() const { return compressionEnabled; }
    GmmResourceUsageType getResourceUsageType();

  protected:
    void applyAuxFlagsForImage(ImageInfo &imgInfo, bool preferCompressed);
    void setupImageResourceParams(ImageInfo &imgInfo, bool preferCompressed);
    MOCKABLE_VIRTUAL bool extraMemoryFlagsRequired();
    void applyExtraMemoryFlags(const StorageInfo &storageInfo);
    void applyDebugOverrides();
    GmmHelper *gmmHelper = nullptr;

    bool compressionEnabled = false;
    bool preferNoCpuAccess = false;
};
} // namespace NEO
