/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QSettings>

#include "src/compat/compat_win64.h"
#include "src/datovka_shared/log/log.h"
#include "src/settings/registry.h"

/*
 * System wide registry entries have precedence. If system-wide and user
 * settings conflict, then system settings should be taken.
 */

/* Policies. */
#define POLICY_REG_SW_ROOT "HKEY_LOCAL_MACHINE\\SOFTWARE\\Policies"

/* System-wide registry root. */
#define SYS_REG_SW_ROOT "HKEY_LOCAL_MACHINE\\Software"

/* User registry root. */
#define USR_REG_SW_ROOT "HKEY_CURRENT_USER\\Software"

/* Application location. */
#define APP_LOC "CZ.NIC\\Datovka"

#define REG_FMT QSettings::NativeFormat

/*
 * Just for convenience. These values should only be printed.
 * They should not be used to access the registry.
 */
#define WOW64_POLICY_REG_SW_ROOT "HKEY_LOCAL_MACHINE\\SOFTWARE\\WOW6432Node\\Policies"
#define WOW64_SYS_REG_SW_ROOT "HKEY_LOCAL_MACHINE\\Software\\WOW6432Node"
#define WOW64_USR_REG_SW_ROOT "HKEY_CURRENT_USER\\Software\\WOW6432Node"

/*
 * 32- and 64- bit registry are separated. The 64-bit regedit displays the
 * 32-bit registry entries in "HKEY_LOCAL_MACHINE\\Software\\WOW6432Node".
 * Datovka for Windows is a 32-bit application, therefore you must use the
 * aforementioned registry path (containing WOW6432Node).
 * https://support.microsoft.com/en-us/help/305097/how-to-view-the-system-registry-by-using-64-bit-versions-of-windows
 */

/*!
 * @brief Converts enum Location into registry root name.
 */
static
QString locationName(enum RegPreferences::Location loc)
{
	switch (loc) {
	case RegPreferences::LOC_POL:
		return POLICY_REG_SW_ROOT;
		break;
	case RegPreferences::LOC_SYS:
		return SYS_REG_SW_ROOT;
		break;
	case RegPreferences::LOC_USR:
		return USR_REG_SW_ROOT;
		break;
	default:
		Q_ASSERT(0);
		return QString();
		break;
	}
}

#if defined(Q_OS_WIN)
/*!
 * @brief Converts enum Location into WOW64 registry root name.
 */
static
QString locationNameWow64(enum RegPreferences::Location loc)
{
	switch (loc) {
	case RegPreferences::LOC_POL:
		return WOW64_POLICY_REG_SW_ROOT;
		break;
	case RegPreferences::LOC_SYS:
		return WOW64_SYS_REG_SW_ROOT;
		break;
	case RegPreferences::LOC_USR:
		return WOW64_USR_REG_SW_ROOT;
		break;
	default:
		Q_ASSERT(0);
		return QString();
		break;
	}
}
#endif /* Q_OS_WIN */

/*!
 * @brief Converts enum Entry into registry name.
 */
static
QString entryName(enum RegPreferences::Entry entry)
{
	switch (entry) {
	case RegPreferences::ENTR_DISABLE_VER_NOTIF:
		return "DisableVersionNotification";
		break;
	case RegPreferences::ENTR_CONF_BASEDIR:
		return "ConfBaseDir";
		break;
	case RegPreferences::ENTR_CONF_SUBDIR:
		return "ConfSubdir";
		break;
	case RegPreferences::ENTR_CONF_DIR:
		return "ConfDir";
		break;
	default:
		Q_ASSERT(0);
		return QString();
		break;
	}
}

bool RegPreferences::haveEntry(enum Location loc, enum Entry entr)
{
	const QString root(locationName(loc));
	if (Q_UNLIKELY(root.isEmpty())) {
		Q_ASSERT(0);
		return false;
	}
	const QString eName(entryName(entr));
	if (Q_UNLIKELY(eName.isEmpty())) {
		Q_ASSERT(0);
		return false;
	}

	QString appRoot(root + "\\" APP_LOC);
	const QSettings settings(appRoot, REG_FMT);

	//bool ret = settings.childKeys().contains(eName, Qt::CaseInsensitive);
	bool ret = settings.contains(eName);

#if defined(Q_OS_WIN)
	if (isWow64()) {
		const QString rootWow64(locationNameWow64(loc));
		if (!rootWow64.isEmpty()) {
			appRoot = rootWow64 + "\\" APP_LOC;
		}
	}
#endif /* Q_OS_WIN */
	logInfoNL("Registry entry '%s': %s",
	    (appRoot + "\\" + eName).toUtf8().constData(),
	    ret ? "present" : "missing");

	return ret;
}

bool RegPreferences::disableVersionNotification(enum Location loc)
{
	if (!haveEntry(loc, ENTR_DISABLE_VER_NOTIF)) {
		return false;
	}
	const QString root(locationName(loc));
	if (Q_UNLIKELY(root.isEmpty())) {
		Q_ASSERT(0);
		return false;
	}
	const QString eName(entryName(ENTR_DISABLE_VER_NOTIF));
	if (Q_UNLIKELY(eName.isEmpty())) {
		Q_ASSERT(0);
		return false;
	}

	QString appRoot(root + "\\" APP_LOC);
	const QSettings settings(appRoot, REG_FMT);

	bool ret = settings.value(eName, false).toBool();

#if defined(Q_OS_WIN)
	if (isWow64()) {
		const QString rootWow64(locationNameWow64(loc));
		Q_ASSERT(!rootWow64.isEmpty());
		appRoot = rootWow64 + "\\" APP_LOC;
	}
#endif /* Q_OS_WIN */
	logInfoNL("Registry entry '%s': %s",
	    (appRoot + "\\" + eName).toUtf8().constData(),
	    ret ? "true" : "false");

	return ret;
}

/*!
 * @brief Return string value held in registry.
 *
 * @param[in] loc Specifies registry location base to search in.
 * @param[in] entr Entry to search for.
 * @return String value if such found.
 */
static
QString registryStringEntry(enum RegPreferences::Location loc,
    enum RegPreferences::Entry entr)
{
	if (!RegPreferences::haveEntry(loc, entr)) {
		return QString();
	}

	const QString root(locationName(loc));
	if (Q_UNLIKELY(root.isEmpty())) {
		Q_ASSERT(0);
		return QString();
	}
	const QString eName(entryName(entr));
	if (Q_UNLIKELY(eName.isEmpty())) {
		Q_ASSERT(0);
		return QString();
	}

	QString appRoot(root + "\\" APP_LOC);
	const QSettings settings(appRoot, REG_FMT);

	QString ret = settings.value(eName, QString()).toString();

#if defined(Q_OS_WIN)
	if (isWow64()) {
		const QString rootWow64(locationNameWow64(loc));
		Q_ASSERT(!rootWow64.isEmpty());
		appRoot = rootWow64 + "\\" APP_LOC;
	}
#endif /* Q_OS_WIN */
	logInfoNL("Registry entry '%s': %s",
	    (appRoot + "\\" + eName).toUtf8().constData(),
	    ret.toUtf8().constData());

	return ret;
}

QString RegPreferences::confBaseDir(enum Location loc)
{
	return registryStringEntry(loc, ENTR_CONF_BASEDIR);
}

QString RegPreferences::confSubdir(enum Location loc)
{
	return registryStringEntry(loc, ENTR_CONF_SUBDIR);
}

QString RegPreferences::confDir(enum Location loc)
{
	return registryStringEntry(loc, ENTR_CONF_DIR);
}
